// src/context/CartContext.jsx
import React, { createContext, useContext, useState, useEffect } from 'react';
import { message } from 'antd';
import axios from 'axios';
import { URL } from "../config";

const CartContext = createContext();

export const useCart = () => {
  const context = useContext(CartContext);
  if (!context) {
    throw new Error('useCart must be used within a CartProvider');
  }
  return context;
};

export const CartProvider = ({ children }) => {
  const [cartItems, setCartItems] = useState([]);
  const [loading, setLoading] = useState(false);
  const [user, setUser] = useState(null);

  // API Base URL
//   const URL = process.env.REACT_APP_API_URL || 'http://localhost:3000';

  // Load user from localStorage on mount
  useEffect(() => {
    const storedUser = localStorage.getItem('user');
    if (storedUser) {
      try {
        const parsedUser = JSON.parse(storedUser);
        setUser(parsedUser);
        // Fetch cart items for this user
        fetchCartItems(parsedUser);
      } catch (error) {
        console.error('Error parsing user from localStorage:', error);
      }
    }
  }, []);

  // Fetch cart items from backend
  const fetchCartItems = async (currentUser = user) => {
    if (!currentUser || !currentUser.buyerId) {
      setCartItems([]);
      return;
    }

    try {
      setLoading(true);
      const endpoint = currentUser.fkUsertypeId === 1 
        ? `/add-to-carts/buyer/${currentUser.buyerId}`
        : `/add-to-carts/seller/${currentUser.sellerId}`;
      
      const response = await axios.get(`${URL}${endpoint}`);
      setCartItems(response.data || []);
    } catch (error) {
      console.error('Error fetching cart items:', error);
      message.error('Failed to load saved properties');
      setCartItems([]);
    } finally {
      setLoading(false);
    }
  };

  // Add item to cart
  const addToCart = async (property) => {
    if (!user) {
      message.warning('Please login to save properties');
      return false;
    }

    try {
      setLoading(true);

      // Check if property already exists in cart
      const existingItem = cartItems.find(
        item => item.fkPostpropertyId === property.postPropertyId
      );

      if (existingItem) {
        message.info('Property already in your saved list');
        return false;
      }

      // Prepare cart data
      const cartData = {
        fkBuyerIdSellerId: user.fkUsertypeId === 1 ? user.buyerId : user.sellerId,
        fkPostpropertyId: property.postPropertyId,
        fkUsertypeId: user.fkUsertypeId,
        date: new Date().toISOString().split('T')[0],
        time: new Date().toTimeString().split(' ')[0],
        lastChanged: new Date().toISOString(),
      };

      // Save to backend
      const response = await axios.post(`${URL}/add-to-carts`, cartData);

      // Refresh cart items
      await fetchCartItems();

      message.success('Property saved successfully');
      return true;
    } catch (error) {
      console.error('Error adding to cart:', error);
      message.error('Failed to save property');
      return false;
    } finally {
      setLoading(false);
    }
  };

  // Remove item from cart
  const removeFromCart = async (cartId) => {
    try {
      setLoading(true);

      await axios.delete(`${URL}/add-to-carts/${cartId}`);

      // Update local state
      setCartItems(prevItems => prevItems.filter(item => item.addTocartId !== cartId));

      message.success('Property removed from saved list');
      return true;
    } catch (error) {
      console.error('Error removing from cart:', error);
      message.error('Failed to remove property');
      return false;
    } finally {
      setLoading(false);
    }
  };

  // Check if property is in cart
  const isInCart = (propertyId) => {
    return cartItems.some(item => item.fkPostpropertyId === propertyId);
  };

  // Get cart count
  const getCartCount = () => {
    return cartItems.length;
  };

  // Clear entire cart
  const clearCart = async () => {
    try {
      setLoading(true);

      // Delete all cart items for current user
      const deletePromises = cartItems.map(item => 
        axios.delete(`${URL}/add-to-carts/${item.addTocartId}`)
      );

      await Promise.all(deletePromises);

      setCartItems([]);
      message.success('All saved properties cleared');
      return true;
    } catch (error) {
      console.error('Error clearing cart:', error);
      message.error('Failed to clear saved properties');
      return false;
    } finally {
      setLoading(false);
    }
  };

  // Update user and fetch their cart
  const updateUser = (newUser) => {
    setUser(newUser);
    if (newUser) {
      localStorage.setItem('user', JSON.stringify(newUser));
      fetchCartItems(newUser);
    } else {
      localStorage.removeItem('user');
      setCartItems([]);
    }
  };

  const value = {
    cartItems,
    loading,
    user,
    addToCart,
    removeFromCart,
    isInCart,
    getCartCount,
    clearCart,
    fetchCartItems,
    updateUser,
  };

  return <CartContext.Provider value={value}>{children}</CartContext.Provider>;
};