import React, { useState, useEffect } from "react";
import { useSearchParams, useNavigate } from "react-router-dom";
import axios from "axios";
import { Row, Col, Typography, Pagination, Tag, Button, Drawer } from "antd";
import { FilterOutlined, SearchOutlined } from "@ant-design/icons";
import HeaderNavbar from "../../Components/Navbar/Header/Header";
import SidebarFilters from "../../Components/PropertyList/SideBarFiltersNew";
import PropertyTags from "../../Components/PropertyList/PropertyTags";
import SortDropdown from "../../Components/PropertyList/SortDropdown";
import PropertyCard from "../../Components/PropertyList/PropertyCard";
import AdBanner from "../../Components/PropertyList/AdBanner2";
import { URL } from "../../config";
import Loading from "../../Components/Loading/Loading";
import "./PropertyListingPage.css";
import Footerone from "../../Components/Footer/Footerone";
import { DeleteOutlined } from "@ant-design/icons";

const { Title } = Typography;

const PropertyListingPage = () => {
  const [searchParams] = useSearchParams();
  const navigate = useNavigate();
  const [propList, setPropList] = useState([]);
  const [filtered, setFiltered] = useState([]);
  const [selectedTags, setSelectedTags] = useState([]);
  const [searchQuery, setSearchQuery] = useState("");
  const [sidebarVisible, setSidebarVisible] = useState(false);
  const [user, setUser] = useState(null);

  // Main filter state that will be shared between components
  const [filters, setFilters] = useState({
    postTypeId: null,
    propertyTypeId: null,
    mainPropertyTypeId: null,
    userId: null,
    priceRange: [0, 100000000],
    propertyTypeIds: [], // Array for multiple property type IDs
    bedrooms: null,
    bathrooms: null,
    areaRange: [0, 5000],
    furnishing: null,
    possession: null,
    parking: null,
    verified: false,
  });

  const [sortOption, setSortOption] = useState(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [sidebarOptions, setSidebarOptions] = useState({
    propertyTypes: [],
    mainProperties: [],
    postTypes: [],
    users: [],
    sellers: [],
  });

  const pageSize = 7;

  // Read URL parameters and set initial filters
  useEffect(() => {
    const search = searchParams.get("search");
    const postTypeId = searchParams.get("postTypeId");
    const budget = searchParams.get("budget");
    const mainPropertyTypeId = searchParams.get("mainPropertyTypeId");
    const propertyTypeIds = searchParams.get("propertyTypeIds");
    const bedrooms = searchParams.get("bedrooms");
    const furnishing = searchParams.get("furnishing");

    const newFilters = { ...filters };

    if (search) {
      setSearchQuery(search);
    }

    // Post Type ID directly from URL
    if (postTypeId) {
      newFilters.postTypeId = parseInt(postTypeId);
    }

    // Main Property Type ID directly from URL
    if (mainPropertyTypeId) {
      newFilters.mainPropertyTypeId = parseInt(mainPropertyTypeId);
    }

    // Property Type IDs - comma separated string to array of numbers
    if (propertyTypeIds) {
      const ids = propertyTypeIds
        .split(",")
        .map((id) => parseInt(id))
        .filter((id) => !isNaN(id));
      newFilters.propertyTypeIds = ids;
    }

    // Budget ranges
    if (budget) {
      const budgetRanges = {
        "0-50": [0, 5000000],
        "50-1": [5000000, 10000000],
        "1-2": [10000000, 20000000],
        "2+": [20000000, 100000000],
      };
      newFilters.priceRange = budgetRanges[budget] || [0, 100000000];
    }

    // Bedrooms
    if (bedrooms) {
      newFilters.bedrooms = parseInt(bedrooms);
    }

    // Furnishing
    if (furnishing) {
      newFilters.furnishing = furnishing;
    }

    setFilters(newFilters);
  }, [searchParams]);

  useEffect(() => {
    const fetchAllData = async () => {
      try {
        const [
          propRes,
          propertyTypeRes,
          mainPropRes,
          postTypeRes,
          userTypeRes,
          sellerRes,
        ] = await Promise.all([
          axios.get(`${URL}/post-property-masters/published-active`),
          axios.get(`${URL}/property-type-masters`),
          axios.get(`${URL}/main-propertty-type-masters`),
          axios.get(`${URL}/post-type-masters`),
          axios.get(`${URL}/usertypes`),
          axios.get(`${URL}/seller-masters`),
        ]);

        setPropList(propRes.data || []);

        setSidebarOptions({
          propertyTypes: propertyTypeRes.data || [],
          mainProperties: mainPropRes.data || [],
          postTypes: postTypeRes.data || [],
          users: (userTypeRes.data || []).filter((u) => u.userId !== 1),
          sellers: sellerRes.data || [],
        });
      } catch (err) {
        console.error("Error fetching data:", err);
      } finally {
        setLoading(false);
      }
    };

    fetchAllData();
  }, []);

  useEffect(() => {
    const storedUser = localStorage.getItem("user");
    if (storedUser) {
      setUser(JSON.parse(storedUser));
    }
  }, []);

  const mapPropertyToUI = (p) => {
    const { users, sellers } = sidebarOptions;
    const userTypeMap = {};
    users.forEach((u) => {
      const type = u.username || u.type || u.role || "";
      userTypeMap[u.userId] = type;
    });

    const tags = [];
    const seller = sellers.find(
      (s) => Number(s.sellerId) === Number(p.fksellerId)
    );

    if (seller && seller.fkuserId && userTypeMap[seller.fkuserId]) {
      const userType = userTypeMap[seller.fkuserId];
      if (userType.toLowerCase().includes("owner")) tags.push("Owner");
      else tags.push(userType);
    }

    if (p.isVerified) tags.push("Verified");
    if (p.availabilityStatus?.toLowerCase() === "ready to move")
      tags.push("Ready to Move");

    return {
      id: p.postpropertyId,
      image: p.photos || "/default.jpg",
      photos: p.photos || [],
      title: p.Property_Title || "No Title",
      location: p.fkcityId || "No Location",
      locality: p.fklocalityId || "No Locality",
      price: Number(p.price) || "Price on request",
      area: p.builduparea ? `${p.builduparea} sqft` : "N/A",
      type: p.fkmainpropertytypeId === 2 ? "Apartment" : "Property",
      fkuserId: seller?.fkuserId ?? null,
      fksellerId: p.fksellerId,
      fkpostTypeId: p.fkpostpropertytypeId,
      fkpropertyTypeId: p.fkpropertyTypeId,
      fkmainpropertytypeId: p.fkmainpropertytypeId,
      createdAt: p.createdAt || null,
      tags,
      desc: p.Property_Description || "No description",
      noofbalkanies: p.noofbalkanies || "No balkanies",
      noofbedrooms: p.noofbedrooms || "No bedrooms",
      noofbathrooms: p.noofbathrooms || "No bathrooms",
      posttypeName: p.posttypeName || "No data",
      possessionType: p.possessionName || "No data",
    };
  };

  // Apply filters including search
  useEffect(() => {
    if (!propList.length) return;

    let result = [...propList];
    const { sellers } = sidebarOptions;

    // Filter only published properties
    result = result.filter((p) => Number(p.publish_Fag) === 2);

    // Search filter
    if (searchQuery && searchQuery.trim()) {
      const query = searchQuery.toLowerCase().trim();
      result = result.filter((p) => {
        const title = (p.Property_Title || "").toLowerCase();
        const location = (p.apartmentNameAreaName || "").toLowerCase();
        const city = (p.fkcityId || "").toLowerCase();
        const locality = (p.fklocalityId || "").toLowerCase();
        const description = (p.Property_Description || "").toLowerCase();

        return (
          title.includes(query) ||
          location.includes(query) ||
          city.includes(query) ||
          locality.includes(query) ||
          description.includes(query)
        );
      });
    }

    // Post Type filter by ID
    if (filters.postTypeId !== null) {
      result = result.filter(
        (p) => Number(p.fkpostpropertytypeId) === filters.postTypeId
      );
    }

    // Main Property Type filter by ID
    if (filters.mainPropertyTypeId !== null) {
      result = result.filter(
        (p) => Number(p.fkmainpropertytypeId) === filters.mainPropertyTypeId
      );
    }

    // Property Type filter - FIXED LOGIC
    // Combine both sidebar selection and URL params
    // const allPropertyTypeIds = [];

    // // Add from URL params (from SearchBarNew)
    // if (filters.propertyTypeIds && filters.propertyTypeIds.length > 0) {
    //   allPropertyTypeIds.push(...filters.propertyTypeIds);
    // }

    // // Add from sidebar selection (for backward compatibility)
    // if (filters.propertyTypeId !== null) {
    //   allPropertyTypeIds.push(filters.propertyTypeId);
    // }

    // // Apply filter if we have any property type IDs
    // if (allPropertyTypeIds.length > 0) {
    //   result = result.filter((p) => {
    //     const propertyTypeId = Number(p.fkpropertyTypeId);
    //     return allPropertyTypeIds.includes(propertyTypeId);
    //   });
    // }
    // Property Type filter - FIXED
    if (filters.propertyTypeIds && filters.propertyTypeIds.length > 0) {
      result = result.filter((p) => {
        const propertyTypeId = Number(p.fkpropertyTypeId);
        return filters.propertyTypeIds.includes(propertyTypeId);
      });
    }

    // User Type filter
    if (filters.userId !== null) {
      const matchingSellers = sellers.filter(
        (s) => Number(s.fkuserId) === filters.userId
      );
      const sellerIds = matchingSellers.map((s) => Number(s.sellerId));
      result = result.filter((p) => sellerIds.includes(Number(p.fksellerId)));
    }

    // Bedrooms filter
    if (filters.bedrooms !== null) {
      result = result.filter(
        (p) => Number(p.noofbedrooms) === filters.bedrooms
      );
    }

    // Bathrooms filter
    if (filters.bathrooms !== null) {
      result = result.filter(
        (p) => Number(p.noofbathrooms) === filters.bathrooms
      );
    }

    // Area range filter
    if (filters.areaRange) {
      const [minArea, maxArea] = filters.areaRange;
      result = result.filter((p) => {
        const area = Number(p.builduparea) || 0;
        return area >= minArea && area <= maxArea;
      });
    }

    // Furnishing filter
    if (filters.furnishing !== null) {
      const furnishingMap = {
        furnished: 1,
        semifurnished: 2,
        unfurnished: 3,
      };
      const furnishingId =
        furnishingMap[filters.furnishing.toLowerCase().replace("-", "")];
      if (furnishingId) {
        result = result.filter(
          (p) => Number(p.fkfurnishingtypeId) === furnishingId
        );
      }
    }

    // Possession filter
    if (filters.possession !== null) {
      const possessionMap = {
        readytomove: 2,
        underconstruction: 1,
      };
      const possessionId =
        possessionMap[filters.possession.toLowerCase().replace(" ", "")];
      if (possessionId) {
        result = result.filter(
          (p) => Number(p.fkpossessionId) === possessionId
        );
      }
    }

    // Parking filter
    if (filters.parking !== null) {
      if (filters.parking === "open") {
        result = result.filter((p) => Number(p.openparking) > 0);
      } else if (filters.parking === "covered") {
        result = result.filter((p) => Number(p.coverparking) > 0);
      } else if (filters.parking === "both") {
        result = result.filter(
          (p) => Number(p.openparking) > 0 && Number(p.coverparking) > 0
        );
      }
    }

    // Verified filter
    if (filters.verified) {
      result = result.filter((p) => p.isVerified);
    }

    // Price range filter - apply BEFORE mapping
    if (filters.priceRange) {
      const [minPrice, maxPrice] = filters.priceRange;
      result = result.filter((p) => {
        const price = Number(p.price) || 0;
        return price >= minPrice && price <= maxPrice;
      });
    }

    // Map to UI format
    let mapped = result.map(mapPropertyToUI);

    // Sort
    if (sortOption === "priceLow")
      mapped = mapped.sort((a, b) => a.price - b.price);
    else if (sortOption === "priceHigh")
      mapped = mapped.sort((a, b) => b.price - a.price);
    else if (sortOption === "newest")
      mapped = mapped.sort(
        (a, b) => new Date(b.createdAt) - new Date(a.createdAt)
      );

    // Tag filter
    const finalList =
      selectedTags.length > 0
        ? mapped.filter((item) =>
            selectedTags.every((t) => item.tags.includes(t))
          )
        : mapped;

    setFiltered(finalList);
    setCurrentPage(1);
  }, [
    filters,
    selectedTags,
    propList,
    sidebarOptions,
    sortOption,
    searchQuery,
  ]);

  // Handle filter changes from SidebarFilters
  const handleSidebarFilter = (vals) => {
    setFilters((prev) => ({
      ...prev,
      postTypeId: vals.postTypeId ?? prev.postTypeId,
      propertyTypeIds: vals.propertyTypeIds ?? prev.propertyTypeIds,
      mainPropertyTypeId: vals.mainPropertyTypeId ?? prev.mainPropertyTypeId,
      userId: vals.userId ?? prev.userId,
      priceRange: vals.priceRange ?? prev.priceRange,
      bedrooms: vals.bedrooms ?? prev.bedrooms,
      bathrooms: vals.bathrooms ?? prev.bathrooms,
      areaRange: vals.areaRange ?? prev.areaRange,
      furnishing: vals.furnishing ?? prev.furnishing,
      possession: vals.possession ?? prev.possession,
      parking: vals.parking ?? prev.parking,
      verified: vals.verified ?? prev.verified,
      propertyTypeIds: vals.propertyTypeIds ?? prev.propertyTypeIds,
    }));
  };

  const paginatedData = filtered.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  // Get active filter labels for display
  const getActiveFilters = () => {
    const active = [];

    // Post Type filter
    if (filters.postTypeId) {
      const postType = sidebarOptions.postTypes.find(
        (pt) => pt.postTypeId === filters.postTypeId
      );
      if (postType)
        active.push({
          key: "postType",
          label: ` ${postType.posttypeName}`,
          color: "blue",
        });
    }

    // Main Property Type filter
    if (filters.mainPropertyTypeId) {
      const mainProp = sidebarOptions.mainProperties.find(
        (mp) => mp.mainpropertytypeId === filters.mainPropertyTypeId
      );
      if (mainProp)
        active.push({
          key: "mainProperty",
          label: `${mainProp.mainprpoertyName}`,
          color: "purple",
        });
    }

    // Property Types filter
    if (filters.propertyTypeIds && filters.propertyTypeIds.length > 0) {
      filters.propertyTypeIds.forEach((typeId) => {
        const propType = sidebarOptions.propertyTypes.find(
          (pt) => pt.propertyTypeId === typeId
        );
        if (propType)
          active.push({
            key: `propertyType-${typeId}`,
            label: propType.propertytypeName,
            color: "orange",
          });
      });
    }
    // Single property type from sidebar (for backward compatibility)
    if (filters.propertyTypeId) {
      const propType = sidebarOptions.propertyTypes.find(
        (pt) => pt.propertyTypeId === filters.propertyTypeId
      );
      if (propType)
        active.push({
          key: "propertyType",
          label: propType.propertytypeName,
          color: "orange",
        });
    }
    // Posted By (User Type)
    if (filters.userId) {
      const user = sidebarOptions.users.find(
        (u) => u.userId === filters.userId
      );
      if (user)
        active.push({
          key: "userId",
          label: ` ${user.username || user.type || user.role}`,
          color: "cyan",
        });
    }

    // Budget filter
    if (
      filters.priceRange &&
      (filters.priceRange[0] !== 0 || filters.priceRange[1] !== 100000000)
    ) {
      const min =
        filters.priceRange[0] >= 10000000
          ? `₹${(filters.priceRange[0] / 10000000).toFixed(1)}Cr`
          : filters.priceRange[0] >= 100000
          ? `₹${(filters.priceRange[0] / 100000).toFixed(0)}L`
          : `₹${filters.priceRange[0]}`;
      const max =
        filters.priceRange[1] >= 10000000
          ? `₹${(filters.priceRange[1] / 10000000).toFixed(1)}Cr`
          : filters.priceRange[1] >= 100000
          ? `₹${(filters.priceRange[1] / 100000).toFixed(0)}L`
          : `₹${filters.priceRange[1]}`;
      active.push({
        key: "budget",
        label: `Budget : ${min} - ${max}`,
        color: "green",
      });
    }

    // Bedrooms filter
    if (filters.bedrooms) {
      active.push({
        key: "bedrooms",
        label: `${filters.bedrooms} BHK`,
        color: "red",
      });
    }

    // Bathrooms filter
    if (filters.bathrooms) {
      active.push({
        key: "bathrooms",
        label: `${filters.bathrooms} Bathroom${
          filters.bathrooms > 1 ? "s" : ""
        }`,
        color: "blue",
      });
    }

    // Area Range filter
    if (
      filters.areaRange &&
      (filters.areaRange[0] !== 0 || filters.areaRange[1] !== 5000)
    ) {
      active.push({
        key: "areaRange",
        label: `Area: ${filters.areaRange[0]} - ${filters.areaRange[1]} sq.ft.`,
        color: "purple",
      });
    }

    // Furnishing filter
    if (filters.furnishing) {
      const furnishingLabel =
        filters.furnishing.charAt(0).toUpperCase() +
        filters.furnishing.slice(1);
      active.push({
        key: "furnishing",
        label: `${furnishingLabel}`,
        color: "lime",
      });
    }

    // Possession filter
    if (filters.possession) {
      const possessionLabel =
        filters.possession === "readytomove"
          ? "Ready to Move"
          : "Under Construction";
      active.push({
        key: "possession",
        label: possessionLabel,
        color: "gold",
      });
    }

    // Parking filter
    if (filters.parking) {
      const parkingLabel =
        filters.parking.charAt(0).toUpperCase() + filters.parking.slice(1);
      active.push({
        key: "parking",
        label: `Parking: ${parkingLabel}`,
        color: "volcano",
      });
    }

    // Verified filter
    if (filters.verified) {
      active.push({
        key: "verified",
        label: "Verified Only",
        color: "success",
      });
    }

    if (filters.verified) {
      active.push({ key: "verified", label: "Verified Only" });
    }

    return active;
  };

  const clearFilter = (filterKey) => {
    if (filterKey === "postType") {
      setFilters((prev) => ({ ...prev, postTypeId: null }));
    } else if (filterKey === "mainProperty") {
      setFilters((prev) => ({ ...prev, mainPropertyTypeId: null }));
    } else if (filterKey === "propertyType") {
      setFilters((prev) => ({ ...prev, propertyTypeId: null }));
    } else if (filterKey.startsWith("propertyType-")) {
      const typeId = parseInt(filterKey.split("-")[1]);
      setFilters((prev) => ({
        ...prev,
        propertyTypeIds: prev.propertyTypeIds.filter((id) => id !== typeId),
      }));
    } else if (filterKey === "userId") {
      setFilters((prev) => ({ ...prev, userId: null }));
    } else if (filterKey === "budget") {
      setFilters((prev) => ({ ...prev, priceRange: [0, 100000000] }));
    } else if (filterKey === "bedrooms") {
      setFilters((prev) => ({ ...prev, bedrooms: null }));
    } else if (filterKey === "bathrooms") {
      setFilters((prev) => ({ ...prev, bathrooms: null }));
    } else if (filterKey === "areaRange") {
      setFilters((prev) => ({ ...prev, areaRange: [0, 5000] }));
    } else if (filterKey === "furnishing") {
      setFilters((prev) => ({ ...prev, furnishing: null }));
    } else if (filterKey === "possession") {
      setFilters((prev) => ({ ...prev, possession: null }));
    } else if (filterKey === "parking") {
      setFilters((prev) => ({ ...prev, parking: null }));
    } else if (filterKey === "verified") {
      setFilters((prev) => ({ ...prev, verified: false }));
    }
  };

  if (loading) return <Loading fullscreen />;

  return (
    <>
      <HeaderNavbar user={user} setUser={setUser} />
      <div
        className="container-fluid"
        style={{ padding: "24px", backgroundColor: "#f4f5f7", marginTop: 40 }}
      >
        <Row gutter={16} style={{ padding: "20px 40px" }}>
          <Col xs={24} sm={24} md={19}>
            <div
              style={{
                display: "flex",
                justifyContent: "space-between",
                alignItems: "center",
                marginBottom: 6,
              }}
            >
              <Title level={4} style={{ fontFamily: "Poppins", marginTop: 10 }}>
                {filtered.length} results | Lello Certified Properties
              </Title>

              <div
                style={{
                  display: "flex",
                  alignItems: "center",
                  gap: 8,
                }}
              >
                <div
                  style={{
                    display: "flex",
                    alignItems: "center",
                    backgroundColor: "#f8f9fa",
                    borderRadius: "50px",
                    padding: "4px 10px 0px 10px",
                    border: "1px solid #e0e0e0",
                    transition: "all 0.3s ease",
                  }}
                  onMouseEnter={(e) => {
                    e.currentTarget.style.borderColor = "#1e40af";
                    e.currentTarget.style.boxShadow =
                      "0 0 0 3px rgba(30, 64, 175, 0.1)";
                  }}
                  onMouseLeave={(e) => {
                    e.currentTarget.style.borderColor = "#e0e0e0";
                    e.currentTarget.style.boxShadow = "none";
                  }}
                >
                  <input
                    type="text"
                    placeholder="Search properties..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    onKeyPress={(e) => {
                      if (e.key === "Enter") {
                        setCurrentPage(1);
                      }
                    }}
                    style={{
                      flex: 1,
                      fontSize: "14px",
                      color: "#1f2937",
                      backgroundColor: "transparent",
                      border: "none",
                      outline: "none",
                      padding: "12px 200px 12px 20px",
                      fontFamily: "Poppins",
                    }}
                  />
                  <Button
                    type="primary"
                    icon={<SearchOutlined />}
                    size="middle"
                    onClick={() => setCurrentPage(1)}
                    style={{
                      backgroundColor: "#1e40af",
                      borderColor: "#1e40af",
                      borderRadius: "50px",
                      height: "38px",
                      padding: "0 20px",
                      display: "flex",
                      alignItems: "center",
                      gap: "8px",
                      fontWeight: 500,
                      transition: "all 0.3s ease",
                      fontFamily: "Poppins",
                    }}
                    onMouseEnter={(e) => {
                      e.currentTarget.style.backgroundColor = "#1e3a8a";
                      e.currentTarget.style.transform = "translateY(-1px)";
                    }}
                    onMouseLeave={(e) => {
                      e.currentTarget.style.backgroundColor = "#1e40af";
                      e.currentTarget.style.transform = "translateY(0)";
                    }}
                  >
                    Search
                  </Button>
                </div>

                <Button
                  type="primary"
                  icon={<FilterOutlined />}
                  onClick={() => setSidebarVisible(true)}
                  style={{
                    display: "flex",
                    alignItems: "center",
                    fontFamily: "Poppins",
                    borderRadius: 50,
                    height: 42,
                    width: 150,
                    fontWeight: 600,
                  }}
                >
                  Filters
                </Button>
              </div>
            </div>

            {/* Active Filters Display */}
            <div
              style={{
                marginBottom: 6,
                display: "flex",
                flexWrap: "wrap",
                gap: 8,
                alignItems: "center",
              }}
            >
              {searchQuery && (
                <Tag
                  color="blue"
                  closable
                  onClose={() => setSearchQuery("")}
                  style={{
                    fontSize: 14,
                    padding: "4px 12px",
                    borderRadius: 50,
                    fontFamily: "Poppins",
                    fontWeight: 500,
                    display: "flex",
                    alignItems: "center",
                    gap: 6,
                  }}
                >
                  Search: {searchQuery}
                </Tag>
              )}
              {getActiveFilters().map((filter) => (
                <Tag
                  key={filter.key}
                  // color={filter.color || "geekblue"}
                  color="geekblue"
                  closable
                  onClose={() => clearFilter(filter.key)}
                  style={{
                    fontSize: 14,
                    padding: "4px 12px",
                    borderRadius: 50,
                    fontFamily: "Poppins",
                    fontWeight: 500,
                    boxShadow: "0 2px 4px rgba(0,0,0,0.08)",
                  }}
                >
                  {filter.label}
                </Tag>
              ))}
              {(searchQuery || getActiveFilters().length > 0) && (
                <Button
                  type="text"
                  size="middle"
                  onClick={() => {
                    setSearchQuery("");
                    setFilters({
                      postTypeId: null,
                      propertyTypeId: null,
                      mainPropertyTypeId: null,
                      userId: null,
                      priceRange: [0, 100000000],
                      propertyTypeIds: [],
                      bedrooms: null,
                      bathrooms: null,
                      areaRange: [0, 5000],
                      furnishing: null,
                      possession: null,
                      parking: null,
                      verified: false,
                    });
                  }}
                  icon={<DeleteOutlined />}
                  style={{
                    display: "flex",
                    alignItems: "center",
                    gap: "6px",
                    padding: "6px 14px",
                    borderRadius: "999px",
                    fontFamily: "Poppins",
                    fontWeight: 600,
                    fontSize: "13px",
                    color: "#dc2626",
                    backgroundColor: "#fee2e2",
                    border: "1px solid #fecaca",
                    transition: "all 0.2s ease",
                  }}
                  onMouseEnter={(e) => {
                    e.currentTarget.style.backgroundColor = "#fecaca";
                  }}
                  onMouseLeave={(e) => {
                    e.currentTarget.style.backgroundColor = "#fee2e2";
                  }}
                >
                  Clear All Filters
                </Button>
              )}
            </div>

            <Row
              justify="space-between"
              align="middle"
              style={{ marginBottom: 16 }}
            >
              {/* Uncomment if needed
              <Col xs={24} md={22}>
                <PropertyTags
                  selectedTags={selectedTags}
                  setSelectedTags={setSelectedTags}
                />
              </Col>
              <Col
                xs={24}
                md={2}
                className="ms-auto"
                style={{ display: "flex", justifyContent: "center" }}
              >
                <SortDropdown onChange={setSortOption} />
              </Col>
              */}
            </Row>

            <Row gutter={[16, 16]}>
              {paginatedData.length > 0 ? (
                paginatedData.map((property) => (
                  <Col span={24} key={property.id}>
                    <PropertyCard property={property} />
                  </Col>
                ))
              ) : (
                <Col span={24}>
                  <div
                    style={{
                      textAlign: "center",
                      padding: "40px",
                      backgroundColor: "white",
                      borderRadius: "8px",
                    }}
                  >
                    <Title level={4}>No properties found</Title>
                    <p>Try adjusting your filters or search query</p>
                  </div>
                </Col>
              )}
            </Row>

            {paginatedData.length > 0 && (
              <Row justify="center" style={{ marginTop: 24 }}>
                <Pagination
                  current={currentPage}
                  pageSize={pageSize}
                  total={filtered.length}
                  onChange={setCurrentPage}
                  showSizeChanger={false}
                />
              </Row>
            )}
          </Col>

          <Col
            xs={0}
            sm={0}
            md={5}
            style={{
              display: "flex",
              justifyItems: "center",
            }}
          >
            <Row>
              <Col xs={24} sm={24} md={24}>
                <AdBanner />
              </Col>
            </Row>
          </Col>
        </Row>
      </div>

      {/* Filter Drawer for mobile/tablet */}
      <Drawer
        placement="left"
        onClose={() => setSidebarVisible(false)}
        open={sidebarVisible}
        width={400}
        closable={false}
        bodyStyle={{
          padding: 0,
          top: "100px",
          zIndex: 1000,
          fontFamily: "Poppins",
        }}
      >
        <SidebarFilters
          onFilterChange={handleSidebarFilter}
          propertyTypes={sidebarOptions.propertyTypes}
          mainProperties={sidebarOptions.mainProperties}
          postTypes={sidebarOptions.postTypes}
          users={sidebarOptions.users}
          activeFilters={filters}
        />
      </Drawer>

      {/* Sidebar for desktop - visible on larger screens */}
      <div className="desktop-sidebar" style={{ display: "none" }}>
        <Col xs={24} sm={8} md={8}>
          <SidebarFilters
            onFilterChange={handleSidebarFilter}
            propertyTypes={sidebarOptions.propertyTypes}
            mainProperties={sidebarOptions.mainProperties}
            postTypes={sidebarOptions.postTypes}
            users={sidebarOptions.users}
            activeFilters={filters}
          />
        </Col>
      </div>
      <Footerone />
    </>
  );
};

export default PropertyListingPage;
