// src/pages/PaymentCallback.js
import React, { useEffect, useState, useRef } from "react";
import { useNavigate, useSearchParams } from "react-router-dom";
import {
  Box,
  Container,
  Typography,
  Button,
  CircularProgress,
  Paper,
} from "@mui/material";
import { CheckCircle, XCircle, Clock, ArrowRight, Home } from "lucide-react";
import {
  checkPaymentStatus,
  verifyPaymentAndCreatePurchase,
} from "../../services/PhonePeService";

function PaymentCallback() {
  const [searchParams] = useSearchParams();
  const navigate = useNavigate();
  const [paymentStatus, setPaymentStatus] = useState("loading");
  const [paymentDetails, setPaymentDetails] = useState(null);
  const [transactionId, setTransactionId] = useState(null);

  const isVerifying = useRef(false);

  useEffect(() => {
    const orderId = searchParams.get("orderId");
    setTransactionId(orderId);
    if (orderId && !isVerifying.current) {
      isVerifying.current = true;
      verifyPayment();
    }
  }, [searchParams]);

  // const verifyPayment = async () => {
  //   try {
  //     const orderId = searchParams.get("orderId");

  //     if (!orderId) {
  //       setPaymentStatus("error");
  //       return;
  //     }

  //     // Payment status तपासा
  //     const response = await checkPaymentStatus(orderId);

  //     if (response.success) {
  //       if (response.status === "COMPLETED") {
  //         setPaymentStatus("success");
  //         setPaymentDetails(response);

  //         // Verify payment and create package purchase
  //         const purchaseResponse = await verifyPaymentAndCreatePurchase(orderId);

  //         if (!purchaseResponse.success) {
  //           console.error("Failed to create package purchase:", purchaseResponse.error);
  //           // Still show success for payment, but log the error
  //         }
  //       } else if (response.status === "FAILED") {
  //         setPaymentStatus("failed");
  //         setPaymentDetails(response);
  //       } else {
  //         setPaymentStatus("pending");
  //         setPaymentDetails(response);
  //       }
  //     } else {
  //       setPaymentStatus("error");
  //     }
  //   } catch (error) {
  //     console.error("Payment verification error:", error);
  //     setPaymentStatus("error");
  //   }
  // };

  const verifyPayment = async () => {
    try {
      const orderId = searchParams.get("orderId");

      if (!orderId) {
        setPaymentStatus("error");
        return;
      }

      // Payment status तपासा
      const response = await checkPaymentStatus(orderId);

      if (response.success) {
        if (response.status === "COMPLETED") {
          setPaymentStatus("success");
          setPaymentDetails(response);

          // Verify payment and create package purchase
          const purchaseResponse =
            await verifyPaymentAndCreatePurchase(orderId);

          if (!purchaseResponse.success) {
            console.error(
              "Failed to create package purchase:",
              purchaseResponse.error,
            );
            // Still show success for payment, but log the error
          }
        } else if (response.status === "FAILED") {
          setPaymentStatus("failed");
          setPaymentDetails(response);
        } else {
          setPaymentStatus("pending");
          setPaymentDetails(response);
        }
      } else {
        setPaymentStatus("error");
      }
    } catch (error) {
      console.error("Payment verification error:", error);
      setPaymentStatus("error");
    } finally {
      isVerifying.current = false;
    }
  };

  const renderContent = () => {
    switch (paymentStatus) {
      case "loading":
        return (
          <Box sx={{ textAlign: "center", py: 8 }}>
            <CircularProgress size={60} sx={{ color: "#00D9FF", mb: 3 }} />
            <Typography variant="h5" sx={{ color: "white", mb: 1 }}>
              Verifying Payment...
            </Typography>
            <Typography sx={{ color: "rgba(255,255,255,0.6)" }}>
              Please wait while we confirm your payment
            </Typography>
          </Box>
        );

      case "success":
        return (
          <Box sx={{ textAlign: "center", py: 6 }}>
            <Box
              sx={{
                width: 100,
                height: 100,
                borderRadius: "50%",
                bgcolor: "rgba(76,175,80,0.1)",
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                margin: "0 auto",
                mb: 3,
              }}
            >
              <CheckCircle size={60} color="#4CAF50" />
            </Box>

            <Typography
              variant="h4"
              sx={{ color: "white", fontWeight: 700, mb: 1 }}
            >
              Payment Successful!
            </Typography>

            <Typography
              variant="h6"
              sx={{ color: "rgba(255,255,255,0.7)", mb: 4 }}
            >
              Welcome to Lello Prime 🎉
            </Typography>

            <Paper
              sx={{
                bgcolor: "rgba(255,255,255,0.05)",
                p: 3,
                borderRadius: 2,
                mb: 4,
              }}
            >
              <Typography sx={{ color: "rgba(255,255,255,0.6)", mb: 1 }}>
                Transaction ID
              </Typography>
              <Typography sx={{ color: "white", fontWeight: 600, mb: 2 }}>
                {paymentDetails?.transactionId || transactionId}
              </Typography>

              <Typography sx={{ color: "rgba(255,255,255,0.6)", mb: 1 }}>
                Amount Paid
              </Typography>
              <Typography
                variant="h5"
                sx={{ color: "#4CAF50", fontWeight: 700 }}
              >
                ₹ {paymentDetails?.amount}
              </Typography>
            </Paper>

            <Box
              sx={{
                display: "flex",
                gap: 2,
                justifyContent: "center",
                flexWrap: "wrap",
              }}
            >
              <Button
                variant="contained"
                startIcon={<Home size={20} />}
                onClick={() => navigate("/")}
                sx={{
                  bgcolor: "#FF6B35",
                  color: "white",
                  px: 4,
                  py: 1.5,
                  textTransform: "none",
                  fontWeight: 600,
                  "&:hover": {
                    bgcolor: "#ff5722",
                  },
                }}
              >
                Go to Home
              </Button>

              <Button
                variant="outlined"
                endIcon={<ArrowRight size={20} />}
                onClick={() => navigate("/properties")}
                sx={{
                  color: "white",
                  borderColor: "rgba(255,255,255,0.3)",
                  px: 4,
                  py: 1.5,
                  textTransform: "none",
                  fontWeight: 600,
                  "&:hover": {
                    borderColor: "rgba(255,255,255,0.5)",
                    bgcolor: "rgba(255,255,255,0.05)",
                  },
                }}
              >
                Browse Properties
              </Button>
            </Box>
          </Box>
        );

      case "failed":
        return (
          <Box sx={{ textAlign: "center", py: 6 }}>
            <Box
              sx={{
                width: 100,
                height: 100,
                borderRadius: "50%",
                bgcolor: "rgba(244,67,54,0.1)",
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                margin: "0 auto",
                mb: 3,
              }}
            >
              <XCircle size={60} color="#f44336" />
            </Box>

            <Typography
              variant="h4"
              sx={{ color: "white", fontWeight: 700, mb: 1 }}
            >
              Payment Failed
            </Typography>

            <Typography
              variant="h6"
              sx={{ color: "rgba(255,255,255,0.7)", mb: 4 }}
            >
              Unfortunately, your payment could not be processed
            </Typography>

            <Paper
              sx={{
                bgcolor: "rgba(244,67,54,0.1)",
                border: "1px solid rgba(244,67,54,0.3)",
                p: 3,
                borderRadius: 2,
                mb: 4,
              }}
            >
              <Typography sx={{ color: "rgba(255,255,255,0.8)" }}>
                {paymentDetails?.message || "Payment was declined or cancelled"}
              </Typography>
            </Paper>

            <Box
              sx={{
                display: "flex",
                gap: 2,
                justifyContent: "center",
                flexWrap: "wrap",
              }}
            >
              <Button
                variant="contained"
                onClick={() => navigate("/membership")}
                sx={{
                  bgcolor: "#FF6B35",
                  color: "white",
                  px: 4,
                  py: 1.5,
                  textTransform: "none",
                  fontWeight: 600,
                  "&:hover": {
                    bgcolor: "#ff5722",
                  },
                }}
              >
                Try Again
              </Button>

              <Button
                variant="outlined"
                onClick={() => navigate("/")}
                sx={{
                  color: "white",
                  borderColor: "rgba(255,255,255,0.3)",
                  px: 4,
                  py: 1.5,
                  textTransform: "none",
                  fontWeight: 600,
                  "&:hover": {
                    borderColor: "rgba(255,255,255,0.5)",
                    bgcolor: "rgba(255,255,255,0.05)",
                  },
                }}
              >
                Go to Home
              </Button>
            </Box>
          </Box>
        );

      case "pending":
        return (
          <Box sx={{ textAlign: "center", py: 6 }}>
            <Box
              sx={{
                width: 100,
                height: 100,
                borderRadius: "50%",
                bgcolor: "rgba(255,152,0,0.1)",
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                margin: "0 auto",
                mb: 3,
              }}
            >
              <Clock size={60} color="#FF9800" />
            </Box>

            <Typography
              variant="h4"
              sx={{ color: "white", fontWeight: 700, mb: 1 }}
            >
              Payment Pending
            </Typography>

            <Typography
              variant="h6"
              sx={{ color: "rgba(255,255,255,0.7)", mb: 4 }}
            >
              Your payment is being processed
            </Typography>

            <Paper
              sx={{
                bgcolor: "rgba(255,152,0,0.1)",
                border: "1px solid rgba(255,152,0,0.3)",
                p: 3,
                borderRadius: 2,
                mb: 4,
              }}
            >
              <Typography sx={{ color: "rgba(255,255,255,0.8)", mb: 2 }}>
                Please wait for the confirmation. You will receive an update
                shortly.
              </Typography>
              <Button
                variant="text"
                onClick={verifyPayment}
                sx={{
                  color: "#FF9800",
                  textTransform: "none",
                }}
              >
                Refresh Status
              </Button>
            </Paper>

            <Button
              variant="outlined"
              onClick={() => navigate("/")}
              sx={{
                color: "white",
                borderColor: "rgba(255,255,255,0.3)",
                px: 4,
                py: 1.5,
                textTransform: "none",
                fontWeight: 600,
                "&:hover": {
                  borderColor: "rgba(255,255,255,0.5)",
                  bgcolor: "rgba(255,255,255,0.05)",
                },
              }}
            >
              Go to Home
            </Button>
          </Box>
        );

      default:
        return (
          <Box sx={{ textAlign: "center", py: 6 }}>
            <XCircle size={60} color="#f44336" />
            <Typography variant="h5" sx={{ color: "white", mt: 2 }}>
              Something went wrong
            </Typography>
            <Button
              variant="contained"
              onClick={() => navigate("/membership")}
              sx={{ mt: 3 }}
            >
              Back to Membership
            </Button>
          </Box>
        );
    }
  };

  return (
    <Box
      sx={{
        minHeight: "100vh",
        background:
          "linear-gradient(135deg, #1a1a2e 0%, #16213e 50%, #0f3460 100%)",
        display: "flex",
        alignItems: "center",
        justifyContent: "center",
        py: 4,
      }}
    >
      <Container maxWidth="md">
        <Paper
          sx={{
            bgcolor: "rgba(255,255,255,0.05)",
            backdropFilter: "blur(10px)",
            borderRadius: 3,
            border: "1px solid rgba(255,255,255,0.1)",
            overflow: "hidden",
          }}
        >
          {renderContent()}
        </Paper>
      </Container>
    </Box>
  );
}

export default PaymentCallback;
