// src/pages/Membership.jsx
// LIGHT THEME VERSION - Updated with #f4b62e and #51aed6 color scheme

import React, { useState, useEffect } from "react";
import {
  Box,
  Container,
  Typography,
  Button,
  Card,
  CardContent,
  Chip,
  Stack,
  IconButton,
  Tooltip,
  useMediaQuery,
  useTheme,
  Grid,
  CircularProgress,
  Alert,
  CardMedia,
  Avatar,
  Badge,
  LinearProgress,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Tab,
  Tabs,
  Divider,
} from "@mui/material";
import {
  Crown,
  Shield,
  Info,
  Check,
  X,
  PhoneCall,
  ArrowRight,
  Star,
  CheckCircle,
  Calendar,
  Clock,
  ShoppingCart,
  History,
  Package,
  Eye,
} from "lucide-react";
import axios from "axios";
import { initiatePhonePePayment } from "../../services/PhonePeService";
import HeaderNavbar from "../../Components/Navbar/Header/Header";
import MembershipConfirmationModal from "../../Components/Modal/MembershipModal/MembershipConfirmationModal";
import { URL } from "../../config";
import { useNavigate } from "react-router-dom";

function Membership() {
  const [selectedPlan, setSelectedPlan] = useState(null);
  const [user, setUser] = useState(null);
  const [openModal, setOpenModal] = useState(false);
  const [packages, setPackages] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [paymentLoading, setPaymentLoading] = useState(false);
  const [activePackage, setActivePackage] = useState(null);
  const [checkingMembership, setCheckingMembership] = useState(false);

  // कार्टमधील प्रॉपर्टीची संख्या
  const [cartCount, setCartCount] = useState(0);
  const [loadingCartCount, setLoadingCartCount] = useState(false);

  // प्रॉपर्टी व्हिजिट संख्या (नवीन)
  const [visitCount, setVisitCount] = useState(0);
  const [loadingVisitCount, setLoadingVisitCount] = useState(false);

  // युजरचे सर्व पॅकेज पर्चेस इतिहास
  const [purchaseHistory, setPurchaseHistory] = useState([]);
  const [loadingHistory, setLoadingHistory] = useState(false);

  // टॅब स्टेट
  const [tabValue, setTabValue] = useState(0);

  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down("md"));
  const isSmallScreen = useMediaQuery(theme.breakpoints.down("sm"));
  const navigate = useNavigate();

  // Check URL parameters for payment success
  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search);
    const paymentSuccess = urlParams.get("paymentSuccess");
    const fromPayment = urlParams.get("fromPayment");

    if (paymentSuccess === "true" || fromPayment === "true") {
      console.log(
        "🎉 Payment success detected! Reloading user's active package...",
      );

      // Clear URL parameters
      window.history.replaceState({}, document.title, window.location.pathname);

      // Force reload active package
      const reloadPackage = async () => {
        if (user) {
          setCheckingMembership(true);
          await checkActiveMembership();
          // Also refresh cart count, visit count and purchase history
          await fetchCartCount();
          await fetchVisitCount();
          await fetchPurchaseHistory();
          setCheckingMembership(false);
        }
      };

      // Wait a bit for localStorage to be updated
      setTimeout(() => {
        reloadPackage();
      }, 500);
    }
  }, [user]);

  // Also add a custom event listener for package purchase
  useEffect(() => {
    const handlePackagePurchased = () => {
      console.log("📦 Package purchased event detected!");
      if (user) {
        checkActiveMembership();
        fetchCartCount();
        fetchVisitCount();
        fetchPurchaseHistory();
      }
    };

    window.addEventListener("packagePurchased", handlePackagePurchased);

    return () => {
      window.removeEventListener("packagePurchased", handlePackagePurchased);
    };
  }, [user]);

  // Fetch saved user from localStorage
  useEffect(() => {
    const storedUser = localStorage.getItem("user");
    console.log("=== Loading User from localStorage ===");
    console.log("Raw stored user:", storedUser);

    if (storedUser) {
      try {
        const parsedUser = JSON.parse(storedUser);
        console.log("Parsed user object:", parsedUser);
        console.log("User ID fields check:", {
          id: parsedUser.id,
          userId: parsedUser.userId,
          sellerId: parsedUser.sellerId,
          buyerId: parsedUser.buyerId,
          fkBuyerIdSellerId: parsedUser.fkBuyerIdSellerId,
        });

        // Check if user has active package
        if (parsedUser.activePackage) {
          const packageEndDate = new Date(parsedUser.activePackage.endDate);
          const currentDate = new Date();

          if (packageEndDate > currentDate) {
            console.log("User has active package:", parsedUser.activePackage);
            setActivePackage(parsedUser.activePackage);
          } else {
            console.log("User's package has expired");
            // Clear expired package from user object
            parsedUser.isPrime = false;
            parsedUser.activePackage = null;
            localStorage.setItem("user", JSON.stringify(parsedUser));
            setUser(parsedUser);
          }
        }

        setUser(parsedUser);
      } catch (error) {
        console.error("Error parsing user:", error);
        alert("Session error. Please log in again.");
      }
    } else {
      console.warn("No user found in localStorage");
      // Optionally redirect to login
      // navigate('/login');
    }
  }, []);

  // Fetch packages from API
  useEffect(() => {
    if (user) {
      fetchPackages();
      // युजर आल्यावर कार्ट काउंट आणि व्हिजिट काउंट मिळवा
      fetchCartCount();
      fetchVisitCount();
      // युजरचा पॅकेज पर्चेस इतिहास मिळवा
      fetchPurchaseHistory();
    }
  }, [user]);

  // Check for active package from server when component mounts
  useEffect(() => {
    if (user && !activePackage) {
      checkActiveMembership();
    }
  }, [user]);

  // फंक्शन: युजरच्या कार्टमधील प्रॉपर्टीची संख्या मिळवा
  const fetchCartCount = async () => {
    try {
      setLoadingCartCount(true);

      // Get user ID
      const userId =
        user?.id ||
        user?.userId ||
        user?.sellerId ||
        user?.buyerId ||
        user?.fkBuyerIdSellerId ||
        user?.fkbuyerId ||
        user?.user_id;

      // Get user type ID
      const userTypeId =
        user?.fkusertypeId || user?.fkuserId || user?.userTypeId;

      if (!userId || !userTypeId) {
        console.error("User ID or UserType ID not found");
        setLoadingCartCount(false);
        return;
      }

      console.log(
        "Fetching cart count for userId:",
        userId,
        "userTypeId:",
        userTypeId,
      );

      // Fetch cart count from server with new API endpoint
      const response = await axios.get(
        `${URL}/add-to-carts/user/${userId}/${userTypeId}/count`,
      );

      if (response.data && response.data.count !== undefined) {
        console.log("Cart count:", response.data.count);
        setCartCount(response.data.count);
      }
    } catch (error) {
      console.error("Error fetching cart count:", error);
      setCartCount(0);
    } finally {
      setLoadingCartCount(false);
    }
  };

  // नवीन फंक्शन: युजरच्या प्रॉपर्टी व्हिजिट संख्या मिळवा
  const fetchVisitCount = async () => {
    try {
      setLoadingVisitCount(true);

      // Get user ID
      const userId =
        user?.id ||
        user?.userId ||
        user?.sellerId ||
        user?.buyerId ||
        user?.fkBuyerIdSellerId ||
        user?.fkbuyerId ||
        user?.user_id;

      // Get user type ID
      const userTypeId =
        user?.fkuserId || user?.fkusertypeId || user?.userTypeId;

      if (!userId || !userTypeId) {
        console.error("User ID or UserType ID not found");
        setLoadingVisitCount(false);
        return;
      }

      console.log(
        "Fetching visit count for userId:",
        userId,
        "userTypeId:",
        userTypeId,
      );

      // Fetch visit count from server with new API endpoint
      const response = await axios.get(
        `${URL}/log-properties/user/${userId}/${userTypeId}/count`,
      );

      if (response.data && response.data.count !== undefined) {
        console.log("Visit count:", response.data.count);
        setVisitCount(response.data.count);
      }
    } catch (error) {
      console.error("Error fetching visit count:", error);
      setVisitCount(0);
    } finally {
      setLoadingVisitCount(false);
    }
  };

  // फंक्शन: युजरचा पॅकेज पर्चेस इतिहास मिळवा
  const fetchPurchaseHistory = async () => {
    try {
      setLoadingHistory(true);

      // Get user ID
      const userId =
        user?.id ||
        user?.userId ||
        user?.sellerId ||
        user?.buyerId ||
        user?.fkBuyerIdSellerId ||
        user?.fkbuyerId ||
        user?.user_id;

      if (!userId) {
        console.error("User ID not found");
        setLoadingHistory(false);
        return;
      }

      console.log("Fetching purchase history for user ID:", userId);

      // Fetch purchase history from server using user-specific endpoint
      const response = await axios.get(
        `${URL}/package-purchases/user/${userId}`,
      );

      console.log("Purchase history response:", response.data);

      if (response.data) {
        // API returns array of purchases for the user
        const userPurchases = Array.isArray(response.data)
          ? response.data
          : [response.data];

        console.log("User purchase history:", userPurchases);
        setPurchaseHistory(userPurchases);
      }
    } catch (error) {
      console.error("Error fetching purchase history:", error);
      setPurchaseHistory([]);
    } finally {
      setLoadingHistory(false);
    }
  };

  const checkActiveMembership = async () => {
    try {
      setCheckingMembership(true);

      // Get user ID
      const userId =
        user?.id ||
        user?.userId ||
        user?.sellerId ||
        user?.buyerId ||
        user?.fkBuyerIdSellerId ||
        user?.fkbuyerId ||
        user?.user_id;

      if (!userId) {
        console.error("User ID not found");
        setCheckingMembership(false);
        return;
      }

      console.log("Checking active membership for user ID:", userId);

      // Fetch user's package purchases from server using correct API
      const response = await axios.get(
        `${URL}/package-purchases/user/${userId}`,
      );

      console.log("Package purchases response:", response.data);

      if (response.data) {
        // response.data हा directly array आहे किंवा object आहे?
        const purchases = Array.isArray(response.data)
          ? response.data
          : [response.data];

        console.log("Parsed purchases:", purchases);

        // Find active package - endDate > current date
        const currentDate = new Date();
        const activePackageData = purchases.find((purchase) => {
          const packageEndDate = new Date(purchase.packageEndDate);
          return packageEndDate > currentDate;
        });

        if (activePackageData) {
          console.log("✅ Found active package:", activePackageData);

          // Update user object with active package
          const updatedUser = {
            ...user,
            isPrime: true,
            activePackage: {
              packageId: activePackageData.fkPackageId,
              packageName:
                activePackageData.package?.packageName || "Premium Package",
              packagePurchaseId: activePackageData.packagePurchaseId,
              startDate: activePackageData.date,
              endDate: activePackageData.packageEndDate,
              transactionId: activePackageData.paymentNo,
              propertyVisit: activePackageData.propertyVisit,
              propertyAddToCart: activePackageData.propertyAddToCart,
            },
          };

          console.log("Updated user with active package:", updatedUser);

          localStorage.setItem("user", JSON.stringify(updatedUser));
          setUser(updatedUser);
          setActivePackage(updatedUser.activePackage);
        } else {
          console.log("❌ No active package found");
        }
      }
    } catch (error) {
      console.error("Error checking active membership:", error);
      console.error("Error details:", error.response?.data);
    } finally {
      setCheckingMembership(false);
    }
  };

  const fetchPackages = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await axios.get(`${URL}/package-masters`);
      const packageData = response.data;

      console.log("All packages from API:", packageData);
      console.log("User userTypeId:", user?.userTypeId);

      // Filter packages based on user's userTypeId
      const filteredPackages = packageData.filter((pkg) => {
        const pkgUserTypeId =
          pkg.fkusertypeId || pkg.userTypeId || pkg.usertype?.userTypeId;
        const userTypeId = user?.fkusertypeId || user?.fkuserId;

        console.log(
          `Package ${pkg.packageName} - fkusertypeId: ${pkgUserTypeId}, User userTypeId: ${userTypeId}`,
        );

        return pkgUserTypeId === userTypeId;
      });

      console.log("Filtered packages for user:", filteredPackages);

      if (filteredPackages.length === 0) {
        setError(
          "Currently, there are no packages available for your user type.",
        );
        setLoading(false);
        return;
      }

      // Sort packages by cost
      const sortedPackages = filteredPackages.sort(
        (a, b) => a.packageCost - b.packageCost,
      );
      setPackages(sortedPackages);

      // Set default selected plan to the middle one
      if (sortedPackages.length > 0) {
        const middleIndex = Math.floor(sortedPackages.length / 2);
        setSelectedPlan(sortedPackages[middleIndex].packageId);
      }

      setLoading(false);
    } catch (error) {
      console.error("Error fetching packages:", error);
      setError("Failed to load membership packages. Please try again later.");
      setLoading(false);
    }
  };

  const handleContinueClick = () => {
    if (!selectedPlan) {
      alert("Please select a package");
      return;
    }
    setOpenModal(true);
  };

  const handleConfirmPayment = async (planDetails) => {
    console.log("=== DEMO Payment Started ===");
    console.log("Plan Details:", planDetails);
    console.log("Current User:", user);

    try {
      setPaymentLoading(true);

      const selectedPackage = packages.find(
        (p) => p.packageId === selectedPlan,
      );

      if (!selectedPackage) {
        throw new Error("Selected package not found");
      }

      console.log("Selected Package:", selectedPackage);

      const userId =
        user?.id ||
        user?.userId ||
        user?.sellerId ||
        user?.buyerId ||
        user?.fkBuyerIdSellerId ||
        user?.fkbuyerId ||
        user?.user_id;

      console.log("User ID:", userId);

      if (!userId) {
        throw new Error("User ID not found. Please log in again.");
      }

      const paymentData = {
        amount: planDetails.price || selectedPackage.packageCost,
        userId: userId,
        userName: user?.name || user?.username || "",
        userPhone: user?.phone || user?.mobile || "",
        userEmail: user?.email || "",
        planName: selectedPackage.packageName,
        packageId: selectedPackage.packageId,
        packageDuration: selectedPackage.packageduration,
        propertyVisit: selectedPackage.propertyVisit,
        propertyAddToCart: selectedPackage.propertyAddToCart,
      };

      console.log("Payment Data:", paymentData);

      const paymentResponse = await initiatePhonePePayment(paymentData);

      console.log("Payment Response:", paymentResponse);

      if (paymentResponse.success) {
        console.log("✅ Demo payment initiated");

        if (paymentResponse.isDemoMode) {
          console.log("Demo mode detected, redirecting to callback...");

          setOpenModal(false);

          setTimeout(() => {
            navigate(
              `/payment/callback?orderId=${paymentResponse.transactionId}`,
            );
            setPaymentLoading(false);
          }, 1000);
        } else {
          window.location.href = paymentResponse.redirectUrl;
        }
      } else {
        throw new Error(paymentResponse.error || "Payment initiation failed");
      }
    } catch (error) {
      console.error("=== Payment Error ===");
      console.error("Error:", error);

      alert(`Payment failed: ${error.message}\n\nPlease try again.`);
      setPaymentLoading(false);
      setOpenModal(false);
    }
  };

  // Get color based on package tier
  const getPackageColor = (index, total) => {
    if (total === 1) return "#51aed6";
    if (index === 0) return "#f4b62e";
    if (index === total - 1) return "#f4b62e";
    return "#51aed6";
  };

  // Get tier name
  const getTierName = (index, total) => {
    if (total === 1) return "Standard";
    if (index === 0) return "Silver";
    if (index === total - 1) return "Diamond";
    return "Gold";
  };

  // Get gradient based on package tier
  const getPackageGradient = (index, total) => {
    if (total === 1) return "linear-gradient(135deg, #51aed6 0%, #3d8bb8 100%)";
    if (index === 0) return "linear-gradient(135deg, #f4b62e 0%, #e0a020 100%)";
    if (index === total - 1)
      return "linear-gradient(135deg, #f4b62e 0%, #e0a020 100%)";
    return "linear-gradient(135deg, #51aed6 0%, #3d8bb8 100%)";
  };

  // Format date for display
  const formatDate = (dateString) => {
    const options = { year: "numeric", month: "long", day: "numeric" };
    return new Date(dateString).toLocaleDateString(undefined, options);
  };

  // Format date and time for display
  const formatDateTime = (dateString, timeString) => {
    const date = new Date(dateString);
    const options = {
      year: "numeric",
      month: "short",
      day: "numeric",
      hour: "2-digit",
      minute: "2-digit",
    };
    return date.toLocaleDateString(undefined, options);
  };

  // Calculate days remaining
  const getDaysRemaining = (endDate) => {
    const today = new Date();
    const expiry = new Date(endDate);
    const diffTime = expiry - today;
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    return diffDays > 0 ? diffDays : 0;
  };

  // Handle tab change
  const handleTabChange = (event, newValue) => {
    setTabValue(newValue);
  };

  // Loading state
  if (loading || !user || checkingMembership) {
    return (
      <>
        <HeaderNavbar user={user} setUser={setUser} />
        <Box
          sx={{
            minHeight: "100vh",
            background: "linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%)",
            display: "flex",
            alignItems: "center",
            justifyContent: "center",
            mt: 8,
          }}
        >
          <Box sx={{ textAlign: "center" }}>
            <CircularProgress size={60} sx={{ color: "#51aed6", mb: 2 }} />
            <Typography variant="h6" sx={{ color: "#333" }}>
              {!user
                ? "Loading User Data..."
                : checkingMembership
                  ? "Checking Membership Status..."
                  : "Loading Packages..."}
            </Typography>
          </Box>
        </Box>
      </>
    );
  }

  // If user has active package, show active membership status
  if (activePackage) {
    const daysRemaining = getDaysRemaining(activePackage.endDate);

    // पॅकेज पर्चेस मधून मिळालेली कार्ट मर्यादा
    const cartLimit = activePackage.propertyAddToCart || 0;
    const cartPercentage =
      cartLimit > 0 ? Math.min((cartCount / cartLimit) * 100, 100) : 0;

    // पॅकेज पर्चेस मधून मिळालेली व्हिजिट मर्यादा (नवीन)
    const visitLimit = activePackage.propertyVisit || 0;
    const visitPercentage =
      visitLimit > 0 ? Math.min((visitCount / visitLimit) * 100, 100) : 0;

    return (
      <>
        <HeaderNavbar user={user} setUser={setUser} />
        <Box
          sx={{
            minHeight: "100vh",
            background: "linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%)",
            py: isSmallScreen ? 2 : 4,
            position: "relative",
            overflow: "hidden",
            mt: 8,
            display: "flex",
            flexDirection: "column",
          }}
        >
          {/* Background pattern overlay */}
          <Box
            sx={{
              position: "absolute",
              top: 0,
              left: 0,
              right: 0,
              bottom: 0,
              backgroundImage: `
                linear-gradient(rgba(255,255,255,0.5) 1px, transparent 1px),
                linear-gradient(90deg, rgba(255,255,255,0.5) 1px, transparent 1px)
              `,
              backgroundSize: "50px 50px",
              opacity: 0.3,
            }}
          />

          <Container
            maxWidth="lg"
            sx={{
              position: "relative",
              zIndex: 1,
              flexGrow: 1,
              display: "flex",
              flexDirection: "column",
            }}
          >
            {/* Header with Tabs */}
            <Box sx={{ mb: 3 }}>
              <Typography
                variant="h4"
                sx={{ color: "#333", fontWeight: 600, mb: 2 }}
              >
                My Membership
              </Typography>

              <Tabs
                value={tabValue}
                onChange={handleTabChange}
                sx={{
                  "& .MuiTabs-indicator": {
                    backgroundColor: "#51aed6",
                  },
                }}
              >
                <Tab
                  label="Active Package"
                  icon={<Package size={18} />}
                  iconPosition="start"
                  sx={{
                    color: "#666",
                    "&.Mui-selected": { color: "#51aed6" },
                  }}
                />
                <Tab
                  label="Purchase History"
                  icon={<History size={18} />}
                  iconPosition="start"
                  sx={{
                    color: "#666",
                    "&.Mui-selected": { color: "#51aed6" },
                  }}
                />
              </Tabs>
              <Divider sx={{ borderColor: "rgba(0,0,0,0.1)" }} />
            </Box>

            {/* Tab Content */}
            {tabValue === 0 && (
              /* Active Membership Card */
              <Card
                sx={{
                  bgcolor: "rgba(81, 174, 214, 0.1)",
                  border: "2px solid rgba(81, 174, 214, 0.3)",
                  borderRadius: 3,
                  overflow: "hidden",
                  mb: 4,
                }}
              >
                <Box
                  sx={{
                    p: 3,
                    background:
                      "linear-gradient(45deg, rgba(81, 174, 214, 0.2) 0%, rgba(61, 139, 184, 0.2) 100%)",
                    borderBottom: "1px solid rgba(0,0,0,0.1)",
                  }}
                >
                  <Box sx={{ display: "flex", alignItems: "center", mb: 2 }}>
                    <Badge
                      overlap="circular"
                      anchorOrigin={{ vertical: "bottom", horizontal: "right" }}
                      badgeContent={
                        <CheckCircle size={24} color="#4CAF50" fill="#4CAF50" />
                      }
                    >
                      <Avatar
                        sx={{
                          bgcolor: "#51aed6",
                          width: 64,
                          height: 64,
                        }}
                      >
                        <Crown size={36} color="#f4b62e" fill="#f4b62e" />
                      </Avatar>
                    </Badge>
                    <Box sx={{ ml: 3 }}>
                      <Typography
                        variant="h4"
                        sx={{ color: "#333", fontWeight: 700 }}
                      >
                        Lello <span style={{ color: "#f4b62e" }}>Prime</span>
                      </Typography>
                      <Typography variant="h6" sx={{ color: "#555" }}>
                        {activePackage.packageName}
                      </Typography>
                    </Box>
                  </Box>
                </Box>

                <CardContent sx={{ p: 3 }}>
                  <Grid container spacing={3}>
                    <Grid item xs={12} sm={6}>
                      <Box
                        sx={{ display: "flex", alignItems: "center", mb: 2 }}
                      >
                        <Calendar size={20} color="#51aed6" />
                        <Typography sx={{ ml: 1, color: "#666" }}>
                          Membership Period
                        </Typography>
                      </Box>
                      <Typography variant="body1" sx={{ color: "#333", mb: 1 }}>
                        {formatDate(activePackage.startDate)} -{" "}
                        {formatDate(activePackage.endDate)}
                      </Typography>
                    </Grid>

                    <Grid item xs={12} sm={6}>
                      <Box
                        sx={{ display: "flex", alignItems: "center", mb: 2 }}
                      >
                        <Clock size={20} color="#51aed6" />
                        <Typography sx={{ ml: 1, color: "#666" }}>
                          Days Remaining
                        </Typography>
                      </Box>
                      <Typography
                        variant="body1"
                        sx={{
                          color: daysRemaining <= 7 ? "#f4b62e" : "#333",
                          fontWeight: 600,
                        }}
                      >
                        {daysRemaining} days
                      </Typography>
                    </Grid>

                    {/* Property Visits - Updated with progress tracking */}
                    <Grid item xs={12} sm={6}>
                      <Box
                        sx={{ display: "flex", alignItems: "center", mb: 2 }}
                      >
                        <Eye size={20} color="#51aed6" />
                        <Typography sx={{ ml: 1, color: "#666" }}>
                          Property Visits
                        </Typography>
                      </Box>
                      <Box>
                        <Box
                          sx={{
                            display: "flex",
                            justifyContent: "space-between",
                            mb: 1,
                          }}
                        >
                          <Typography variant="body1" sx={{ color: "#333" }}>
                            {visitCount} / {visitLimit} visits
                          </Typography>
                          {loadingVisitCount && (
                            <CircularProgress
                              size={16}
                              sx={{ color: "#51aed6" }}
                            />
                          )}
                        </Box>

                        {/* प्रोग्रेस बार */}
                        <Box
                          sx={{
                            width: "100%",
                            bgcolor: "rgba(0,0,0,0.1)",
                            borderRadius: 1,
                            height: 8,
                            mb: 1,
                            overflow: "hidden",
                          }}
                        >
                          <LinearProgress
                            variant="determinate"
                            value={visitPercentage}
                            sx={{
                              height: "100%",
                              borderRadius: 1,
                              backgroundColor: "rgba(0,0,0,0.1)",
                              "& .MuiLinearProgress-bar": {
                                backgroundColor:
                                  visitPercentage >= 100
                                    ? "#f44336"
                                    : "#4CAF50",
                              },
                            }}
                          />
                        </Box>

                        <Typography variant="caption" sx={{ color: "#666" }}>
                          {visitPercentage >= 100
                            ? "You've used all your visits!"
                            : `${Math.round(visitPercentage)}% of your visits used`}
                        </Typography>
                      </Box>
                    </Grid>

                    {/* Add to Cart - Existing with progress tracking */}
                    <Grid item xs={12} sm={6}>
                      <Box
                        sx={{ display: "flex", alignItems: "center", mb: 2 }}
                      >
                        <ShoppingCart size={20} color="#51aed6" />
                        <Typography sx={{ ml: 1, color: "#666" }}>
                          Add to Cart Limit
                        </Typography>
                      </Box>
                      <Box>
                        <Box
                          sx={{
                            display: "flex",
                            justifyContent: "space-between",
                            mb: 1,
                          }}
                        >
                          <Typography variant="body1" sx={{ color: "#333" }}>
                            {cartCount} / {cartLimit} properties
                          </Typography>
                          {loadingCartCount && (
                            <CircularProgress
                              size={16}
                              sx={{ color: "#51aed6" }}
                            />
                          )}
                        </Box>

                        {/* प्रोग्रेस बार */}
                        <Box
                          sx={{
                            width: "100%",
                            bgcolor: "rgba(0,0,0,0.1)",
                            borderRadius: 1,
                            height: 8,
                            mb: 1,
                            overflow: "hidden",
                          }}
                        >
                          <LinearProgress
                            variant="determinate"
                            value={cartPercentage}
                            sx={{
                              height: "100%",
                              borderRadius: 1,
                              backgroundColor: "rgba(0,0,0,0.1)",
                              "& .MuiLinearProgress-bar": {
                                backgroundColor:
                                  cartPercentage >= 100 ? "#f44336" : "#4CAF50",
                              },
                            }}
                          />
                        </Box>

                        <Typography variant="caption" sx={{ color: "#666" }}>
                          {cartPercentage >= 100
                            ? "You've reached your cart limit!"
                            : `${Math.round(cartPercentage)}% of your cart limit used`}
                        </Typography>
                      </Box>
                    </Grid>
                  </Grid>

                  {/* Alerts */}
                  {daysRemaining <= 7 && (
                    <Alert
                      severity="warning"
                      sx={{ mt: 3, bgcolor: "rgba(244, 182, 46, 0.1)" }}
                    >
                      <Typography sx={{ color: "#333" }}>
                        Your membership will expire in {daysRemaining} days.
                        Renew now to continue enjoying Lello Prime benefits!
                      </Typography>
                    </Alert>
                  )}

                  {visitPercentage >= 100 && (
                    <Alert
                      severity="error"
                      sx={{ mt: 2, bgcolor: "rgba(244, 67, 54, 0.1)" }}
                    >
                      <Typography sx={{ color: "#333" }}>
                        You've used all {visitLimit} property visits! To visit
                        more properties, consider upgrading your membership.
                      </Typography>
                    </Alert>
                  )}

                  {cartPercentage >= 100 && (
                    <Alert
                      severity="error"
                      sx={{ mt: 2, bgcolor: "rgba(244, 67, 54, 0.1)" }}
                    >
                      <Typography sx={{ color: "#333" }}>
                        You've reached your cart limit of {cartLimit}{" "}
                        properties! To add more properties, consider upgrading
                        your membership.
                      </Typography>
                    </Alert>
                  )}

                  <Box
                    sx={{ display: "flex", gap: 2, mt: 3, flexWrap: "wrap" }}
                  >
                    {daysRemaining <= 7 && (
                      <Button
                        variant="outlined"
                        onClick={() => setActivePackage(null)}
                        sx={{
                          color: "#51aed6",
                          borderColor: "#51aed6",
                          px: 4,
                          py: 1.5,
                          fontWeight: 600,
                          borderRadius: 2,
                          textTransform: "none",
                          "&:hover": {
                            borderColor: "#3d8bb8",
                            bgcolor: "rgba(81, 174, 214, 0.05)",
                          },
                        }}
                      >
                        Renew Membership
                      </Button>
                    )}
                  </Box>
                </CardContent>
              </Card>
            )}

            {tabValue === 1 && (
              /* Purchase History */
              <Card
                sx={{
                  bgcolor: "rgba(255,255,255,0.8)",
                  border: "1px solid rgba(0,0,0,0.1)",
                  borderRadius: 3,
                  overflow: "hidden",
                  mb: 4,
                }}
              >
                <Box
                  sx={{
                    p: 3,
                    background:
                      "linear-gradient(45deg, rgba(81, 174, 214, 0.1) 0%, rgba(61, 139, 184, 0.1) 100%)",
                    borderBottom: "1px solid rgba(0,0,0,0.1)",
                  }}
                >
                  <Box sx={{ display: "flex", alignItems: "center" }}>
                    <History size={24} color="#51aed6" />
                    <Typography
                      variant="h5"
                      sx={{ color: "#333", fontWeight: 600, ml: 2 }}
                    >
                      Purchase History
                    </Typography>
                  </Box>
                </Box>

                <CardContent sx={{ p: 0 }}>
                  {loadingHistory ? (
                    <Box
                      sx={{ display: "flex", justifyContent: "center", p: 4 }}
                    >
                      <CircularProgress sx={{ color: "#51aed6" }} />
                    </Box>
                  ) : purchaseHistory.length > 0 ? (
                    <TableContainer component={Box} sx={{ maxHeight: 400 }}>
                      <Table stickyHeader>
                        <TableHead>
                          <TableRow>
                            <TableCell
                              sx={{
                                color: "#666",
                                bgcolor: "rgba(81, 174, 214, 0.1)",
                              }}
                            >
                              Purchase ID
                            </TableCell>
                            <TableCell
                              sx={{
                                color: "#666",
                                bgcolor: "rgba(81, 174, 214, 0.1)",
                              }}
                            >
                              Package
                            </TableCell>
                            <TableCell
                              sx={{
                                color: "#666",
                                bgcolor: "rgba(81, 174, 214, 0.1)",
                              }}
                            >
                              Amount
                            </TableCell>
                            <TableCell
                              sx={{
                                color: "#666",
                                bgcolor: "rgba(81, 174, 214, 0.1)",
                              }}
                            >
                              Purchase Date
                            </TableCell>
                            <TableCell
                              sx={{
                                color: "#666",
                                bgcolor: "rgba(81, 174, 214, 0.1)",
                              }}
                            >
                              Valid Until
                            </TableCell>
                            <TableCell
                              sx={{
                                color: "#666",
                                bgcolor: "rgba(81, 174, 214, 0.1)",
                              }}
                            >
                              Status
                            </TableCell>
                          </TableRow>
                        </TableHead>
                        <TableBody>
                          {purchaseHistory.map((purchase) => {
                            const isExpired =
                              new Date(purchase.packageEndDate) < new Date();
                            const isActive =
                              purchase.packagePurchaseId ===
                              activePackage?.packagePurchaseId;

                            return (
                              <TableRow
                                key={purchase.packagePurchaseId}
                                sx={{
                                  "&:nth-of-type(odd)": {
                                    bgcolor: "rgba(0,0,0,0.02)",
                                  },
                                  "&:hover": {
                                    bgcolor: "rgba(81, 174, 214, 0.05)",
                                  },
                                }}
                              >
                                <TableCell
                                  sx={{
                                    color: "#333",
                                    borderBottom: "1px solid rgba(0,0,0,0.05)",
                                  }}
                                >
                                  #{purchase.packagePurchaseId}
                                </TableCell>
                                <TableCell
                                  sx={{
                                    color: "#333",
                                    borderBottom: "1px solid rgba(0,0,0,0.05)",
                                  }}
                                >
                                  <Box>
                                    <Typography
                                      variant="body2"
                                      sx={{ fontWeight: 600 }}
                                    >
                                      {purchase.package?.packageName ||
                                        `Package #${purchase.fkPackageId}`}
                                    </Typography>
                                    <Typography
                                      variant="caption"
                                      sx={{ color: "#666" }}
                                    >
                                      {purchase.propertyVisit} visits,{" "}
                                      {purchase.propertyAddToCart} cart limit
                                    </Typography>
                                  </Box>
                                </TableCell>
                                <TableCell
                                  sx={{
                                    color: "#333",
                                    borderBottom: "1px solid rgba(0,0,0,0.05)",
                                  }}
                                >
                                  ₹{" "}
                                  {purchase.paymentAmount?.toLocaleString(
                                    "en-IN",
                                  )}
                                </TableCell>
                                <TableCell
                                  sx={{
                                    color: "#333",
                                    borderBottom: "1px solid rgba(0,0,0,0.05)",
                                  }}
                                >
                                  {formatDateTime(purchase.date, purchase.time)}
                                </TableCell>
                                <TableCell
                                  sx={{
                                    color: "#333",
                                    borderBottom: "1px solid rgba(0,0,0,0.05)",
                                  }}
                                >
                                  {formatDate(purchase.packageEndDate)}
                                </TableCell>
                                <TableCell
                                  sx={{
                                    borderBottom: "1px solid rgba(0,0,0,0.05)",
                                  }}
                                >
                                  {isActive ? (
                                    <Chip
                                      label="Active"
                                      size="small"
                                      color="success"
                                      sx={{
                                        bgcolor: "rgba(76, 175, 80, 0.1)",
                                        color: "#4CAF50",
                                      }}
                                    />
                                  ) : isExpired ? (
                                    <Chip
                                      label="Expired"
                                      size="small"
                                      color="error"
                                      sx={{
                                        bgcolor: "rgba(244, 67, 54, 0.1)",
                                        color: "#f44336",
                                      }}
                                    />
                                  ) : (
                                    <Chip
                                      label="Completed"
                                      size="small"
                                      color="primary"
                                      sx={{
                                        bgcolor: "rgba(81, 174, 214, 0.1)",
                                        color: "#51aed6",
                                      }}
                                    />
                                  )}
                                </TableCell>
                              </TableRow>
                            );
                          })}
                        </TableBody>
                      </Table>
                    </TableContainer>
                  ) : (
                    <Box sx={{ p: 4, textAlign: "center" }}>
                      <Typography variant="body1" sx={{ color: "#666" }}>
                        No purchase history found.
                      </Typography>
                    </Box>
                  )}
                </CardContent>
              </Card>
            )}
          </Container>
        </Box>
      </>
    );
  }

  // Error state
  if (error) {
    return (
      <>
        <HeaderNavbar user={user} setUser={setUser} />
        <Box
          sx={{
            minHeight: "100vh",
            background: "linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%)",
            display: "flex",
            alignItems: "center",
            justifyContent: "center",
            mt: 8,
            px: 2,
          }}
        >
          <Alert severity="error" sx={{ maxWidth: 600 }}>
            {error}
          </Alert>
        </Box>
      </>
    );
  }

  return (
    <>
      <HeaderNavbar user={user} setUser={setUser} />
      <Box
        sx={{
          minHeight: "100vh",
          background: "linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%)",
          py: isSmallScreen ? 2 : 4,
          position: "relative",
          overflow: "hidden",
          mt: 8,
          display: "flex",
          flexDirection: "column",
        }}
      >
        {/* Background pattern overlay */}
        <Box
          sx={{
            position: "absolute",
            top: 0,
            left: 0,
            right: 0,
            bottom: 0,
            backgroundImage: `
              linear-gradient(rgba(255,255,255,0.5) 1px, transparent 1px),
              linear-gradient(90deg, rgba(255,255,255,0.5) 1px, transparent 1px)
            `,
            backgroundSize: "50px 50px",
            opacity: 0.3,
          }}
        />

        <Container
          maxWidth="lg"
          sx={{
            position: "relative",
            zIndex: 1,
            flexGrow: 1,
            display: "flex",
            flexDirection: "column",
          }}
        >
          {/* Header Section */}
          <Box sx={{ textAlign: "center", mb: isSmallScreen ? 2 : 3 }}>
            <Box sx={{ display: "flex", justifyContent: "center", mb: 1 }}>
              <Crown
                size={isSmallScreen ? 32 : 48}
                color="#f4b62e"
                fill="#f4b62e"
              />
            </Box>

            <Typography
              variant={isSmallScreen ? "h4" : "h3"}
              sx={{
                color: "#333",
                fontWeight: 700,
                mb: 1,
              }}
            >
              Lello <span style={{ color: "#f4b62e" }}>Prime</span>
            </Typography>

            <Typography
              variant={isSmallScreen ? "body1" : "h6"}
              sx={{
                color: "#666",
                mb: 0,
                fontWeight: 400,
              }}
            >
              Pay Zero Commission & Call Owners Directly
            </Typography>
          </Box>

          {/* Pricing Cards */}
          <Box
            sx={{
              flexGrow: 1,
              display: "flex",
              flexDirection: "column",
              justifyContent: "center",
              mb: 2,
            }}
          >
            <Grid
              container
              spacing={isSmallScreen ? 2 : 3}
              justifyContent="center"
            >
              {packages.map((pkg, index) => {
                const tierName = getTierName(index, packages.length);
                const tierColor = getPackageColor(index, packages.length);
                const tierGradient = getPackageGradient(index, packages.length);
                const isRecommended = index === Math.floor(packages.length / 2);
                const isSelected = selectedPlan === pkg.packageId;

                return (
                  <Grid
                    item
                    xs={12}
                    sm={packages.length <= 2 ? 6 : 6}
                    md={packages.length <= 2 ? 6 : 4}
                    key={pkg.packageId}
                    sx={{ display: "flex", justifyContent: "center" }}
                  >
                    <Card
                      onClick={() => setSelectedPlan(pkg.packageId)}
                      sx={{
                        width: "100%",
                        maxWidth: isSmallScreen ? "100%" : "320px",
                        bgcolor: isSelected
                          ? "rgba(81, 174, 214, 0.15)"
                          : "rgba(255,255,255,0.8)",
                        border: isSelected
                          ? `2px solid ${tierColor}`
                          : "1px solid rgba(0,0,0,0.1)",
                        borderRadius: 3,
                        cursor: "pointer",
                        position: "relative",
                        transition: "all 0.3s ease",
                        height: "100%",
                        display: "flex",
                        flexDirection: "column",
                        overflow: "hidden",
                        transform: isSelected ? "scale(1.02)" : "scale(1)",
                        boxShadow: isSelected
                          ? `0 10px 30px ${tierColor}40, 0 0 0 1px ${tierColor}30 inset`
                          : "0 4px 20px rgba(0,0,0,0.1)",
                        "&:hover": {
                          transform: isSmallScreen
                            ? "none"
                            : "translateY(-8px) scale(1.02)",
                          boxShadow: `0 12px 30px ${tierColor}40`,
                          bgcolor: isSelected
                            ? "rgba(81, 174, 214, 0.2)"
                            : "rgba(255,255,255,0.9)",
                        },
                      }}
                    >
                      {isRecommended && (
                        <Box
                          sx={{
                            position: "absolute",
                            top: -5,
                            right: -5,
                            background: tierGradient,
                            color: "white",
                            py: 0.5,
                            px: 1.5,
                            textAlign: "center",
                            fontSize: "12px",
                            fontWeight: 700,
                            zIndex: 2,
                            display: "flex",
                            alignItems: "center",
                            justifyContent: "center",
                            gap: 0.5,
                            borderRadius: "0 0 0 12px",
                            boxShadow: "0 2px 8px rgba(0,0,0,0.2)",
                          }}
                        >
                          <Star size={12} fill="white" />
                          RECOMMENDED
                          <Star size={12} fill="white" />
                        </Box>
                      )}

                      {/* Package Photo */}
                      <Box
                        sx={{
                          position: "relative",
                          height: isSmallScreen ? 380 : 420,
                          overflow: "hidden",
                        }}
                      >
                        {pkg.packagePhoto && (
                          <img
                            src={pkg.packagePhoto}
                            alt={pkg.packageName}
                            style={{
                              width: "100%",
                              height: "100%",
                              objectFit: "cover",
                            }}
                          />
                        )}

                        <Box
                          sx={{
                            position: "absolute",
                            top: 0,
                            left: 0,
                            right: 0,
                            bottom: 0,
                          }}
                        />

                        <Box
                          sx={{
                            position: "absolute",
                            top: 0,
                            left: 0,
                            right: 0,
                            bottom: 0,
                            padding: isSmallScreen ? 2 : 3,
                            display: "flex",
                            flexDirection: "column",
                            zIndex: 1,
                          }}
                        >
                          <Box
                            sx={{
                              display: "flex",
                              justifyContent: "end",
                              alignItems: "flex-start",
                              mt: isRecommended ? 4 : 2,
                            }}
                          >
                            {isSelected && (
                              <Avatar
                                sx={{
                                  bgcolor: tierColor,
                                  width: 32,
                                  height: 32,
                                }}
                              >
                                <CheckCircle size={20} />
                              </Avatar>
                            )}
                          </Box>
                        </Box>
                      </Box>

                      {/* Package Details */}
                      <Box
                        sx={{
                          p: 2,
                          flexGrow: 1,
                          display: "flex",
                          flexDirection: "column",
                        }}
                      >
                        <Typography
                          variant="h6"
                          sx={{ color: "#333", fontWeight: 600, mb: 1 }}
                        >
                          {pkg.packageName}
                        </Typography>

                        <Typography
                          variant="h5"
                          sx={{ color: tierColor, fontWeight: 700, mb: 2 }}
                        >
                          ₹ {pkg.packageCost}
                        </Typography>

                        <Box sx={{ mb: 2, flexGrow: 1 }}>
                          {/* Property Visits */}
                          <Box
                            sx={{
                              display: "flex",
                              alignItems: "center",
                              mb: 1,
                            }}
                          >
                            <Eye size={16} color="#51aed6" />
                            <Typography
                              sx={{ ml: 1, color: "#333", fontSize: "14px" }}
                            >
                              {pkg.propertyVisit || 0} Property Visits
                            </Typography>
                          </Box>

                          {/* वर्तमान व्हिजिट काउंट दाखवा */}
                          {!loadingVisitCount && activePackage && (
                            <Box sx={{ mt: 0.5, pl: 3, mb: 1 }}>
                              <Typography
                                variant="caption"
                                sx={{ color: "#666" }}
                              >
                                You currently have {visitCount} visits used
                              </Typography>
                              {pkg.propertyVisit > visitCount && (
                                <Typography
                                  variant="caption"
                                  sx={{ color: "#4CAF50", display: "block" }}
                                >
                                  You can visit {pkg.propertyVisit - visitCount}{" "}
                                  more properties with this plan
                                </Typography>
                              )}
                            </Box>
                          )}

                          {/* Add to Cart */}
                          <Box
                            sx={{
                              display: "flex",
                              alignItems: "center",
                              mb: 1,
                            }}
                          >
                            <ShoppingCart size={16} color="#51aed6" />
                            <Typography
                              sx={{ ml: 1, color: "#333", fontSize: "14px" }}
                            >
                              {pkg.propertyAddToCart || 0} Add to Cart Limit
                            </Typography>
                          </Box>

                          {/* वर्तमान कार्ट काउंट दाखवा */}
                          {!loadingCartCount && activePackage && (
                            <Box sx={{ mt: 0.5, pl: 3 }}>
                              <Typography
                                variant="caption"
                                sx={{ color: "#666" }}
                              >
                                You currently have {cartCount} properties in
                                your cart
                              </Typography>
                              {pkg.propertyAddToCart > cartCount && (
                                <Typography
                                  variant="caption"
                                  sx={{ color: "#4CAF50", display: "block" }}
                                >
                                  You can add{" "}
                                  {pkg.propertyAddToCart - cartCount} more
                                  properties with this plan
                                </Typography>
                              )}
                            </Box>
                          )}
                        </Box>

                        <Typography variant="body2" sx={{ color: "#666" }}>
                          Valid for {pkg.packageduration} days
                        </Typography>
                      </Box>

                      {/* Select Button */}
                      <Button
                        variant="contained"
                        sx={{
                          background: tierGradient,
                          color: "white",
                          fontWeight: 600,
                          py: 1.5,
                          borderRadius: 0,
                          textTransform: "none",
                          boxShadow: "0 4px 10px rgba(0,0,0,0.2)",
                          "&:hover": {
                            background: tierGradient,
                            boxShadow: `0 6px 12px ${tierColor}50`,
                          },
                        }}
                        onClick={(e) => {
                          e.stopPropagation();
                          setSelectedPlan(pkg.packageId);
                        }}
                      >
                        {isSelected ? "Selected" : "Select Plan"}
                      </Button>
                    </Card>
                  </Grid>
                );
              })}
            </Grid>
          </Box>

          {/* CTA Button */}
          <Box sx={{ display: "flex", justifyContent: "center", mb: 2 }}>
            <Button
              variant="contained"
              endIcon={<ArrowRight size={20} />}
              onClick={handleContinueClick}
              disabled={!selectedPlan || paymentLoading}
              sx={{
                background: "linear-gradient(45deg, #f4b62e 0%, #e0a020 100%)",
                color: "white",
                px: isSmallScreen ? 4 : 6,
                py: isSmallScreen ? 1.5 : 2,
                fontSize: isSmallScreen ? "16px" : "18px",
                fontWeight: 600,
                borderRadius: 2,
                textTransform: "none",
                boxShadow: "0 4px 15px rgba(244, 182, 46, 0.4)",
                "&:hover": {
                  background:
                    "linear-gradient(45deg, #e0a020 0%, #c99010 100%)",
                  transform: isSmallScreen ? "none" : "scale(1.05)",
                  boxShadow: "0 6px 20px rgba(244, 182, 46, 0.6)",
                },
                "&:disabled": {
                  background: "rgba(244, 182, 46, 0.3)",
                  color: "rgba(255,255,255,0.5)",
                },
              }}
            >
              {paymentLoading ? (
                <>
                  <CircularProgress size={20} sx={{ mr: 1, color: "white" }} />
                  Processing...
                </>
              ) : (
                <>
                  Continue with{" "}
                  {packages.find((p) => p.packageId === selectedPlan)
                    ?.packageName || "Package"}
                </>
              )}
            </Button>
          </Box>
        </Container>

        {/* Contact Info */}
        <Box
          sx={{
            position: "fixed",
            bottom: isSmallScreen ? 16 : 24,
            right: isSmallScreen ? 16 : 24,
            bgcolor: "rgba(255,255,255,0.9)",
            borderRadius: 3,
            p: isSmallScreen ? 1.5 : 2,
            display: "flex",
            alignItems: "center",
            gap: 2,
            zIndex: 10,
            boxShadow: "0 4px 20px rgba(0,0,0,0.1)",
            backdropFilter: "blur(10px)",
          }}
        >
          <PhoneCall size={isSmallScreen ? 20 : 24} color="#51aed6" />
          <Box>
            <Typography
              sx={{
                color: "#666",
                fontSize: isSmallScreen ? "10px" : "12px",
              }}
            >
              Query for packages?
            </Typography>
            <Typography
              sx={{
                color: "#333",
                fontWeight: 600,
                fontSize: isSmallScreen ? "14px" : "16px",
              }}
            >
              +91-9999999900
            </Typography>
          </Box>
        </Box>
      </Box>

      {/* Confirmation Modal */}
      <MembershipConfirmationModal
        open={openModal}
        onClose={() => setOpenModal(false)}
        selectedPlan={selectedPlan}
        packages={packages}
        onConfirm={handleConfirmPayment}
        loading={paymentLoading}
      />
    </>
  );
}

export default Membership;
