import React, { useEffect, useState } from "react";
import {
  Card,
  Typography,
  Avatar,
  Descriptions,
  Button,
  Space,
  Divider,
  Spin,
} from "antd";
import {
  EditOutlined,
  MailOutlined,
  PhoneOutlined,
  LockOutlined,
  ApartmentOutlined,
  UserOutlined,
  GlobalOutlined,
  LinkedinOutlined,
  HomeOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import { useNavigate } from "react-router-dom";
import axios from "axios";
import coverPhoto from "../../assets/Header/HeroIMG4.jpg";
import { URL } from "../../config";
import Loading from "../../Components/Loading/Loading";
import profilePhoto from "../../assets/images/profile image 2.jpg";

const { Title, Text } = Typography;

const Profile = () => {
  const navigate = useNavigate();
  const [sellerData, setSellerData] = useState(null);
  const [username, setUsername] = useState(null);
  const [cityName, setCityName] = useState(null);
  const [loading, setLoading] = useState(true);
  const [windowWidth, setWindowWidth] = useState(window.innerWidth);

  const getProfilePhotoUrl = () => {
    if (sellerData?.sellerProfile && sellerData.sellerProfile.trim() !== "") {
      return `${URL}/uploads/seller-profiles/${sellerData.sellerProfile}`;
    }
    return profilePhoto; // Default photo
  };

  useEffect(() => {
    const handleUpdate = (e) => {
      if (e.detail) {
        // Update sellerData directly with new values
        setSellerData((prev) => ({
          ...prev,
          sellerName: e.detail.name || prev?.sellerName,
          sellerProfile: e.detail.photo
            ? e.detail.photo.split("/").pop()
            : prev?.sellerProfile,
          fkuserId: e.detail.userId || prev?.fkuserId,
        }));
      }
    };

    window.addEventListener("sellerInfoUpdated", handleUpdate);
    return () => window.removeEventListener("sellerInfoUpdated", handleUpdate);
  }, []);

  useEffect(() => {
    const fetchSellerData = async () => {
      try {
        const sellerId = localStorage.getItem("sellerId");
        if (!sellerId) {
          console.warn("No sellerId found in localStorage");
          return;
        }

        const url = `${URL}/seller-masters/${sellerId}`;
        const response = await axios.get(url);
        setSellerData(response.data);

        if (response.data.fkuserId) {
          const userRes = await axios.get(
            `${URL}/usertypes/${response.data.fkuserId}`
          );
          setUsername(userRes.data.username);
        }

        if (response.data.fkcityId) {
          const cityRes = await axios.get(
            `${URL}/city-masters/${response.data.fkcityId}`
          );
          setCityName(cityRes.data.cityName);
        }
      } catch (error) {
        console.error("Error fetching seller, user, or city info:", error);
      }
    };

    fetchSellerData();
  }, []);

  useEffect(() => {
    const timer = setTimeout(() => {
      setLoading(false);
    }, 1000);
    return () => clearTimeout(timer);
  }, []);

  useEffect(() => {
    const handleResize = () => setWindowWidth(window.innerWidth);
    window.addEventListener("resize", handleResize);
    return () => window.removeEventListener("resize", handleResize);
  }, []);

  if (loading) return <Loading fullscreen />;

  const get = (field) => sellerData?.[field] || "-";
  const avatarSize = windowWidth < 768 ? 150 : 300;

  // Add this just before return statement
  const formatDateTime = (dateString) => {
    if (!dateString) return "-";
    const options = {
      day: "2-digit",
      month: "short",
      year: "numeric",
      hour: "2-digit",
      minute: "2-digit",
      hour12: true,
    };
    return new Date(dateString).toLocaleString("en-IN", options);
  };

  return (
    <div
      style={{
        margin: 10,
        minHeight: "88vh",
        backgroundColor: "#fff",
        borderRadius: 12,
        boxShadow: "0 4px 12px rgba(0, 0, 0, 0.05)",
        overflow: "hidden",
      }}
    >
      {/* Cover */}
      <div
        style={{
          backgroundImage: `url(${coverPhoto})`,
          backgroundSize: "cover",
          backgroundPosition: "center",
          height: 240,
        }}
      ></div>

      {/* Header */}
      <div style={{ padding: 16, marginTop: -200 }}>
        <Space
          style={{
            display: "flex",
            justifyContent: "space-between",
            width: "100%",
            flexWrap: "wrap",
          }}
        >
          <Space
            style={{
              flexWrap: "wrap",
              alignItems: "flex-end",
              marginTop: windowWidth < 768 ? 110 : 0,
            }}
          >
            <Avatar
              size={avatarSize}
              style={{
                border: "5px solid #fff",
                backgroundColor: "#1976d2",
                marginLeft: 12,
              }}
              // src={`${URL}/uploads/seller-profiles/${get("sellerProfile")}`}
              src={getProfilePhotoUrl()}
              icon={!sellerData?.sellerProfile && <UserOutlined />}
              onError={(e) => {
                e.target.onerror = null;
                e.target.src = profilePhoto; // Error वर default photo
              }}
              alt="Profile"
              // onError={(e) => {
              //   e.target.onerror = null;
              //   e.target.src = "https://via.placeholder.com/172?text=No+Image";
              // }}
            />
            <div style={{ marginLeft: 30, marginBottom: 40 }}>
              <Title
                level={2}
                className="text-poppins"
                style={{ marginBottom: 4 }}
              >
                {get("sellerName")}
              </Title>
              <Text
                type="secondary"
                className="text-poppins"
                style={{ fontSize: 14 }}
              >
                {username || "Seller"}
              </Text>
            </div>
          </Space>

          <Button
            className="text-poppins"
            style={{ marginTop: windowWidth < 768 ? 0 : 130 }}
            icon={<EditOutlined />}
            type="primary"
            onClick={() => navigate("/dashboard/edit-profile")}
          >
            Edit Profile
          </Button>
        </Space>

        <Divider />

        {/* Personal & Contact */}
        <Card
          title="Personal & Contact Info"
          bordered={false}
          className="text-poppins"
        >
          <Descriptions column={{ xs: 1, sm: 2 }} className="text-poppins">
            <Descriptions.Item label={<UserOutlined />}>
              <span>{sellerData?.gender?.genderName || "-"}</span>
            </Descriptions.Item>
            <Descriptions.Item label={<PhoneOutlined />}>
              <span>{get("sellerMobile")}</span>
            </Descriptions.Item>
            <Descriptions.Item label={<MailOutlined />}>
              <span>{get("sellerGmail")}</span>
            </Descriptions.Item>
            <Descriptions.Item label="City">
              <span>{cityName || "-"}</span>
            </Descriptions.Item>
            <Descriptions.Item label={<ApartmentOutlined />}>
              <span>{get("sellerLocation")}</span>
            </Descriptions.Item>
            <Descriptions.Item label={<IdcardOutlined />}>
              <span>{get("sellerDesignation")}</span>
            </Descriptions.Item>
            <Descriptions.Item label="Permanent Address">
              <span>{get("sellerPermenentadd")}</span>
            </Descriptions.Item>
            <Descriptions.Item label="Temporary Address">
              <span>{get("sellertempaddress")}</span>
            </Descriptions.Item>
          </Descriptions>
        </Card>

        {/* About Section */}
        <Card
          title="About"
          bordered={false}
          className="text-poppins"
          style={{ marginTop: 16 }}
        >
          <Text className="text-poppins">{get("selleraboutus")}</Text>
        </Card>

        {/* Company Details */}
        <Card
          title="Company Info"
          bordered={false}
          className="text-poppins"
          style={{ marginTop: 16 }}
        >
          <Descriptions column={{ xs: 1, sm: 2 }} className="text-poppins">
            <Descriptions.Item label="Company Name">
              {get("sellercompName")}
            </Descriptions.Item>
            <Descriptions.Item label="Location">
              {get("sellercopmlocation")}
            </Descriptions.Item>
            <Descriptions.Item label="Company Mobile">
              {get("sellercompmobile")}
            </Descriptions.Item>
            <Descriptions.Item label="Company Gmail">
              {get("sellercopmgmail")}
            </Descriptions.Item>
            <Descriptions.Item label="Manager Name">
              {get("sellercopmManegername")}
            </Descriptions.Item>
            <Descriptions.Item label="Manager Mobile">
              {get("sellercopmManegermob")}
            </Descriptions.Item>
            <Descriptions.Item label={<GlobalOutlined />}>
              {get("sellercompwebsite")}
            </Descriptions.Item>
            <Descriptions.Item label={<LinkedinOutlined />}>
              {get("sellerlinkedin")}
            </Descriptions.Item>
            <Descriptions.Item label={<HomeOutlined />}>
              {get("sellerfacebook")}
            </Descriptions.Item>
          </Descriptions>
        </Card>

        {/* Package Info */}
        <Card
          title="Package Info"
          bordered={false}
          className="text-poppins"
          style={{ marginTop: 16 }}
        >
          <Descriptions column={{ xs: 1, sm: 2 }} className="text-poppins">
            <Descriptions.Item label="Package ID">
              {get("fkpackageId")}
            </Descriptions.Item>
            <Descriptions.Item label="Payment Flag">
              {get("paymentFlag")}
            </Descriptions.Item>
            <Descriptions.Item label="Registration Date">
              {formatDateTime(get("regdate"))}
            </Descriptions.Item>
            <Descriptions.Item label="Package Start Date">
              {formatDateTime(get("packagestartdate"))}
            </Descriptions.Item>
            <Descriptions.Item label="Package End Date">
              {formatDateTime(get("packageenddate"))}
            </Descriptions.Item>
          </Descriptions>
        </Card>

        {/* Login Info */}
        <Card
          title="Login Info"
          bordered={false}
          className="text-poppins"
          style={{ marginTop: 16 }}
        >
          <Descriptions column={{ xs: 1, sm: 2 }} className="text-poppins">
            <Descriptions.Item label="User ID">
              {get("fkuserId")}
            </Descriptions.Item>
            <Descriptions.Item label={<LockOutlined />}>
              <span>********</span>
            </Descriptions.Item>
          </Descriptions>
        </Card>
      </div>
    </div>
  );
};

export default Profile;
