/* eslint-disable no-unused-vars */
import React, { useState, useEffect } from "react";
import {
  Card,
  Typography,
  Input,
  Button,
  Upload,
  message,
  Form,
  Row,
  Col,
  Divider,
  Spin,
  Select,
} from "antd";
import {
  SaveOutlined,
  MailOutlined,
  PhoneOutlined,
  UserOutlined,
  ApartmentOutlined,
  GlobalOutlined,
  LinkedinOutlined,
  HomeOutlined,
} from "@ant-design/icons";
import axios from "axios";
import { toast } from "react-toastify";
import { useNavigate } from "react-router-dom";
import { URL } from "../../config";
import Loading from "../../Components/Loading/Loading";

const { Title } = Typography;
const { Option } = Select;

const EditProfile = () => {
  const [form] = Form.useForm();
  const [profileImage, setProfileImage] = useState([]);
  const [loading, setLoading] = useState(true);
  const [sellerId, setSellerId] = useState(null);
  const [users, setUsers] = useState([]);
  const [selectedUserId, setSelectedUserId] = useState(null);

  const [cities, setCities] = useState([]);
  const [selectedCityId, setSelectedCityId] = useState(null);

  const [genders, setGenders] = useState([]);
  const [selectedGenderId, setSelectedGenderId] = useState(null);
  const [isPhotoDeleted, setIsPhotoDeleted] = useState(false);
  const navigate = useNavigate();

  useEffect(() => {
    const id = localStorage.getItem("sellerId");
    if (!id) {
      toast.error("No seller ID found in localStorage");
      return;
    }
    setSellerId(id);
  }, []);

  useEffect(() => {
    const fetchUsers = async () => {
      try {
        const res = await axios.get(`${URL}/usertypes`);
        setUsers(res.data);
      } catch (err) {
        toast.error("Failed to fetch user list");
      }
    };
    fetchUsers();
  }, []);

  useEffect(() => {
    const fetchgender = async () => {
      try {
        const genderRes = await axios.get(`${URL}/gender-masters`);
        console.log(genderRes.data);
        setGenders(genderRes.data);
      } catch (error) {
        console.log(error);
      }
    };
    const fetchCities = async () => {
      try {
        const res = await axios.get(`${URL}/city-masters`);
        setCities(res.data);
        console.log(res.data);
      } catch (err) {
        toast.error("Failed to fetch cities");
      }
    };
    fetchgender();
    fetchCities();
  }, []);

  useEffect(() => {
    if (!sellerId) return;

    const fetchProfile = async () => {
      try {
        const res = await axios.get(`${URL}/seller-masters/${sellerId}`);
        const seller = res.data;

        form.setFieldsValue({
          sellerName: seller.sellerName,
          sellerGmail: seller.sellerGmail,
          sellerMobile: seller.sellerMobile,
          sellerLocation: seller.sellerLocation,
          sellerDesignation: seller.sellerDesignation,
          // sellerGender: seller.sellerGender,
          sellerPermenentadd: seller.sellerPermenentadd,
          sellertempaddress: seller.sellertempaddress,
          selleraboutus: seller.selleraboutus,
          sellercompName: seller.sellercompName,
          sellercopmlocation: seller.sellercopmlocation,
          sellercompmobile: seller.sellercompmobile,
          sellercopmgmail: seller.sellercopmgmail,
          sellercopmManegername: seller.sellercopmManegername,
          sellercopmManegermob: seller.sellercopmManegermob,
          sellercompwebsite: seller.sellercompwebsite,
          sellerfacebook: seller.sellerfacebook,
          sellerlinkedin: seller.sellerlinkedin,
        });

        if (seller.fkuserId) {
          setSelectedUserId(seller.fkuserId);
        }

        if (seller.fkcityId) {
          setSelectedCityId(seller.fkcityId);
        }
        if (seller.fkgenderId) setSelectedGenderId(seller.fkgenderId); // ✅ set genderId

        if (seller.sellerProfile) {
          setProfileImage([
            {
              uid: "-1",
              name: seller.sellerProfile,
              status: "done",
              url: `${URL}/uploads/seller-profiles/${seller.sellerProfile}`,
            },
          ]);
        }

        setLoading(false);
      } catch (error) {
        message.error("Failed to fetch seller profile");
        setLoading(false);
      }
    };

    fetchProfile();
  }, [sellerId, form]);

  const handlePhotoChange = ({ fileList }) => {
    setProfileImage(fileList);

    if (fileList.length === 0) {
      setIsPhotoDeleted(true); // Photo delete केला
    } else {
      setIsPhotoDeleted(false);
    }
  };

  const handleFinish = async (values) => {
    try {
      const formData = new FormData();

      Object.entries(values).forEach(([key, val]) => {
        formData.append(key, val || "");
      });

      if (selectedUserId) {
        formData.append("fkuserId", selectedUserId);
      }

      if (selectedCityId) {
        formData.append("fkcityId", selectedCityId);
      }

      // if (profileImage.length > 0 && profileImage[0].originFileObj) {
      //   formData.append("sellerProfile", profileImage[0].originFileObj);
      // }
      if (isPhotoDeleted) {
        formData.append("sellerProfile", ""); // Database मधून delete होईल
      } else if (profileImage.length > 0 && profileImage[0].originFileObj) {
        formData.append("sellerProfile", profileImage[0].originFileObj);
      }

      if (selectedGenderId) {
        formData.append("fkgenderId", selectedGenderId);
      }

      formData.append("lastChanged", new Date().toISOString());
      formData.append("isActive", 1);

      await axios.patch(`${URL}/seller-masters/${sellerId}`, formData, {
        headers: {
          "Content-Type": "multipart/form-data",
        },
      });

      const updatedSeller = {
        name: values.sellerName,
        company: users.find((u) => u.userId === selectedUserId)?.username || "",
        photo:
          profileImage.length > 0
            ? `${URL}/uploads/seller-profiles/${
                profileImage[0].name
              }?t=${Date.now()}`
            : null,
      };

      localStorage.setItem("sellerInfo", JSON.stringify(updatedSeller));

      // ✅ Custom event dispatch करा
      window.dispatchEvent(
        new CustomEvent("sellerInfoUpdated", { detail: updatedSeller })
      );

      toast.success("Profile updated successfully!");
      navigate("/dashboard/profile");
    } catch (error) {
      console.error("Upload failed", error);
      toast.error("Failed to update profile");
    }
  };

  if (loading) return <Loading fullscreen />;

  return (
    <div style={{ padding: "0 12px" }}>
      <Card
        className="text-poppins"
        style={{
          maxWidth: 1200,
          margin: "0 auto",
          borderRadius: 16,
          boxShadow: "0 6px 24px rgba(0, 0, 0, 0.06)",
          marginTop: "8px",
        }}
      >
        <Title
          className="text-poppins"
          level={3}
          style={{ textAlign: "center", marginBottom: 32 }}
        >
          Edit Seller Profile
        </Title>

        <Form layout="vertical" form={form} onFinish={handleFinish}>
          <Row gutter={[24, 24]}>
            {/* Profile Photo Section */}
            <Col xs={24} md={8}>
              <Card
                title={<span className="text-poppins">Profile Photo</span>}
                bordered={false}
                style={{ borderRadius: 12 }}
              >
                <Upload
                  listType="picture-circle"
                  fileList={profileImage}
                  onChange={({ fileList }) => setProfileImage(fileList)}
                  onRemove={() => {
                    setIsPhotoDeleted(true);
                    return true;
                  }}
                  beforeUpload={() => false}
                  maxCount={1}
                  accept="image/*"
                >
                  {profileImage.length < 1 && "+ Upload"}
                </Upload>
              </Card>
            </Col>

            {/* Personal Details */}
            <Col xs={24} md={16}>
              <Card bordered={false} style={{ borderRadius: 12 }}>
                <Row gutter={16}>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="sellerName"
                      label={<span className="text-poppins">Full Name</span>}
                      rules={[{ required: true }]}
                    >
                      <Input
                        prefix={<UserOutlined />}
                        className="text-poppins"
                      />
                    </Form.Item>
                  </Col>

                  <Col xs={24} sm={12}>
                    <Form.Item
                      label={<span className="text-poppins">User Name</span>}
                    >
                      <Select
                        placeholder="Select user"
                        className="text-poppins"
                        value={selectedUserId}
                        onChange={(value) => setSelectedUserId(value)}
                      >
                        {users.map((user) => (
                          <Option key={user.userId} value={user.userId}>
                            {user.username}
                          </Option>
                        ))}
                      </Select>
                    </Form.Item>
                  </Col>

                  <Col xs={24} sm={12}>
                    <Form.Item
                      label={<span className="text-poppins">City</span>}
                      rules={[
                        {
                          required: true,
                          message: "Please Select the City",
                        },
                      ]}
                    >
                      <Select
                        placeholder="Select city"
                        className="text-poppins"
                        value={selectedCityId}
                        onChange={(value) => setSelectedCityId(value)}
                        showSearch
                        optionFilterProp="children"
                      >
                        {cities.map((city) => (
                          <Option key={city.cityId} value={city.cityId}>
                            {city.cityName}
                          </Option>
                        ))}
                      </Select>
                    </Form.Item>
                  </Col>

                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="sellerDesignation"
                      label={<span className="text-poppins">Designation</span>}
                    >
                      <Input className="text-poppins" />
                    </Form.Item>
                  </Col>

                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="sellerGender"
                      label={<span className="text-poppins">Gender</span>}
                    >
                      <Select
                        placeholder="Select gender"
                        className="text-poppins"
                        value={selectedGenderId}
                        onChange={(value) => setSelectedGenderId(value)}
                      >
                        {genders.map((g) => (
                          <Option key={g.genderId} value={g.genderId}>
                            {g.genderName}
                          </Option>
                        ))}
                      </Select>
                    </Form.Item>
                  </Col>

                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="sellerGmail"
                      label={<span className="text-poppins">Email</span>}
                      rules={[
                        {
                          required: true,
                          message: "Please enter email address",
                        },
                        {
                          type: "email",
                          message: "Please enter a valid email address",
                        },
                      ]}
                    >
                      <Input
                        prefix={<MailOutlined />}
                        className="text-poppins"
                      />
                    </Form.Item>
                  </Col>

                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="sellerMobile"
                      label={<span className="text-poppins">Mobile</span>}
                      rules={[
                        {
                          required: true,
                          message: "Please enter mobile number",
                        },
                        {
                          pattern: /^[0-9]{10}$/,
                          message: "Mobile number must be exactly 10 digits",
                        },
                      ]}
                    >
                      <Input
                        prefix={<PhoneOutlined />}
                        maxLength={10}
                        className="text-poppins"
                        onKeyPress={(e) => {
                          if (!/[0-9]/.test(e.key)) {
                            e.preventDefault();
                          }
                        }}
                      />
                    </Form.Item>
                  </Col>

                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="sellerLocation"
                      label={<span className="text-poppins">Location</span>}
                    >
                      <Input
                        prefix={<ApartmentOutlined />}
                        className="text-poppins"
                      />
                    </Form.Item>
                  </Col>

                  <Col xs={24}>
                    <Form.Item
                      name="sellerPermenentadd"
                      label={
                        <span className="text-poppins">Permanent Address</span>
                      }
                    >
                      <Input.TextArea rows={2} className="text-poppins" />
                    </Form.Item>
                  </Col>

                  <Col xs={24}>
                    <Form.Item
                      name="sellertempaddress"
                      label={
                        <span className="text-poppins">Temporary Address</span>
                      }
                    >
                      <Input.TextArea rows={2} className="text-poppins" />
                    </Form.Item>
                  </Col>

                  <Col xs={24}>
                    <Form.Item
                      name="selleraboutus"
                      label={<span className="text-poppins">About Seller</span>}
                    >
                      <Input.TextArea rows={3} className="text-poppins" />
                    </Form.Item>
                  </Col>
                </Row>
              </Card>
            </Col>

            {/* Company Details */}
            <Col xs={24}>
              <Divider />
              <Title level={5} className="text-poppins">
                Company Details
              </Title>
              <Row gutter={16}>
                <Col xs={24} sm={12}>
                  <Form.Item
                    name="sellercompName"
                    label={<span className="text-poppins">Company Name</span>}
                  >
                    <Input className="text-poppins" />
                  </Form.Item>
                </Col>

                <Col xs={24} sm={12}>
                  <Form.Item
                    name="sellercopmlocation"
                    label={
                      <span className="text-poppins">Company Location</span>
                    }
                  >
                    <Input className="text-poppins" />
                  </Form.Item>
                </Col>

                <Col xs={24} sm={12}>
                  {/* <Form.Item
                    name="sellercompmobile"
                    label={<span className="text-poppins">Company Mobile</span>}
                  >
                    <Input className="text-poppins" />
                  </Form.Item> */}
                  <Form.Item
                    name="sellercompmobile"
                    label={<span className="text-poppins">Company Mobile</span>}
                    rules={[
                      // {
                      //   required: true,
                      //   message: "Please enter mobile number",
                      // },
                      {
                        pattern: /^[0-9]{10}$/,
                        message: "Mobile number must be exactly 10 digits",
                      },
                    ]}
                  >
                    <Input
                      prefix={<PhoneOutlined />}
                      maxLength={10}
                      className="text-poppins"
                      // onKeyPress={(e) => {
                      //   if (!/[0-9]/.test(e.key)) {
                      //     e.preventDefault();
                      //   }
                      // }}
                    />
                  </Form.Item>
                </Col>

                <Col xs={24} sm={12}>
                  <Form.Item
                    name="sellercopmgmail"
                    label={<span className="text-poppins">Company Email</span>}
                    rules={[
                      {
                        type: "email",
                        message: "Please enter a valid email address",
                      },
                    ]}
                  >
                    <Input className="text-poppins" />
                  </Form.Item>
                </Col>

                <Col xs={24} sm={12}>
                  <Form.Item
                    name="sellercopmManegername"
                    label={<span className="text-poppins">Manager Name</span>}
                  >
                    <Input className="text-poppins" />
                  </Form.Item>
                </Col>

                <Col xs={24} sm={12}>
                  {/* <Form.Item
                    name="sellercopmManegermob"
                    label={<span className="text-poppins">Manager Mobile</span>}
                  >
                    <Input className="text-poppins" />
                  </Form.Item> */}
                  <Form.Item
                    name="sellercopmManegermob"
                    label={<span className="text-poppins">Manager Mobile</span>}
                    rules={[
                      // {
                      //   required: true,
                      //   message: "Please enter mobile number",
                      // },
                      {
                        pattern: /^[0-9]{10}$/,
                        message: "Mobile number must be exactly 10 digits",
                      },
                    ]}
                  >
                    <Input
                      prefix={<PhoneOutlined />}
                      maxLength={10}
                      className="text-poppins"
                      // onKeyPress={(e) => {
                      //   if (!/[0-9]/.test(e.key)) {
                      //     e.preventDefault();
                      //   }
                      // }}
                    />
                  </Form.Item>
                </Col>
              </Row>
            </Col>

            {/* Social Links */}
            <Col xs={24}>
              <Divider />
              <Title level={5} className="text-poppins">
                Social Links
              </Title>
              <Row gutter={16}>
                <Col xs={24} sm={12}>
                  <Form.Item
                    name="sellercompwebsite"
                    label={
                      <span className="text-poppins">Company Website</span>
                    }
                    rules={[
                      {
                        type: "url",
                        message:
                          "Please enter a valid website URL (e.g. https://example.com)",
                      },
                    ]}
                  >
                    <Input
                      prefix={<GlobalOutlined />}
                      className="text-poppins"
                      placeholder="https://example.com"
                    />
                  </Form.Item>
                </Col>

                <Col xs={24} sm={12}>
                  <Form.Item
                    name="sellerfacebook"
                    label={<span className="text-poppins">Facebook</span>}
                    rules={[
                      {
                        type: "url",
                        message: "Please enter a valid Facebook URL",
                      },
                      {
                        pattern:
                          /^(https?:\/\/)?(www\.)?facebook\.com\/[A-Za-z0-9_.-]+$/,
                        message:
                          "Facebook URL must be in format https://facebook.com/username",
                      },
                    ]}
                  >
                    <Input
                      prefix={<HomeOutlined />}
                      className="text-poppins"
                      placeholder="https://facebook.com/username"
                    />
                  </Form.Item>
                </Col>

                <Col xs={24} sm={12}>
                  <Form.Item
                    name="sellerlinkedin"
                    label={<span className="text-poppins">LinkedIn</span>}
                    rules={[
                      {
                        type: "url",
                        message: "Please enter a valid URL",
                      },
                      {
                        pattern:
                          /^(https?:\/\/)?(www\.)?linkedin\.com\/(in|company)\/[A-Za-z0-9_-]+\/?$/,
                        message:
                          "LinkedIn URL must be in format https://www.linkedin.com/in/username",
                      },
                    ]}
                  >
                    <Input
                      prefix={<LinkedinOutlined />}
                      className="text-poppins"
                      placeholder="https://www.linkedin.com/in/username"
                    />
                  </Form.Item>
                </Col>
              </Row>
            </Col>
          </Row>

          <Divider />

          {/* Save Button */}
          <div style={{ textAlign: "center" }}>
            <Button
              type="primary"
              htmlType="submit"
              size="large"
              icon={<SaveOutlined />}
              style={{ borderRadius: 8, paddingInline: 32 }}
              className="text-poppins"
            >
              Save Changes
            </Button>
          </div>
        </Form>
      </Card>
    </div>
  );
};

export default EditProfile;
