import React, { useState, useEffect } from "react";
import {
  Form,
  Input,
  Select,
  Button,
  Upload,
  DatePicker,
  Row,
  Col,
} from "antd";
import { UploadOutlined, CheckCircleOutlined } from "@ant-design/icons";
import ArrowBackIcon from "@mui/icons-material/ArrowBack";
import { Box, Paper, Typography } from "@mui/material";
import axios from "axios";
import { toast } from "react-toastify";
import { useParams, useNavigate } from "react-router-dom";
import dayjs from "dayjs";
import Ads from "../../assets/Ads/Ad-1.jpg";
import { URL } from "../../config";
import Loading from "../../Components/Loading/Loading";

const AdsForm = () => {
  const { adsId } = useParams();
  const navigate = useNavigate();
  const [form] = Form.useForm();
  const [loading, setLoading] = useState(false);
  const [publishTypes, setPublishTypes] = useState([]);
  const [fileList, setFileList] = useState([]);

  useEffect(() => {
    fetchPublishTypes();
    if (adsId) {
      fetchAdsDetails(adsId);
    }
  }, [adsId]);

  // Load dropdown values
  const fetchPublishTypes = async () => {
    try {
      setLoading(true);
      const res = await axios.get(`${URL}/ads-type-masters`);
      setPublishTypes(res.data);
    } catch (error) {
      toast.error("Failed to load publish types!");
    }finally {
    setLoading(false); 
  }
  };

  // Load data in update mode
  const fetchAdsDetails = async (id) => {
  try {
    setLoading(true);
    const res = await axios.get(`${URL}/ads-managements/${id}`);
    const data = res.data;

    form.setFieldsValue({
      adsTitle: data.adsTitle,
      adsStartDate: data.adsStartDate ? dayjs(data.adsStartDate) : null,
      adsEndDate: data.adsEndDate ? dayjs(data.adsEndDate) : null,
      fkadsTypeId: data.fkadsTypeId,
    });

    if (data.adsPhoto) {
      setFileList([
        {
          uid: "-1",
          name: data.adsPhoto.split('/').pop(),
          status: "done",
          url: data.adsPhoto.startsWith('http')
            ? data.adsPhoto
            : `${data.adsPhoto.startsWith('/') ? data.adsPhoto : '/' + data.adsPhoto}`,
        },
      ]);
    }
  } catch (error) {
    toast.error("Failed to load ad details!");
  }finally {
    setLoading(false); 
  }
};


  // Save / Update ad
  // Save / Update ad
  const onFinish = async (values) => {
    const sellerId = localStorage.getItem("sellerId"); // or JSON.parse(localStorage.getItem("user")).fksellerId

    if (!sellerId) {
      toast.error("Seller not logged in!");
      return;
    }

    const formData = new FormData();
    formData.append("adsTitle", values.adsTitle);
    formData.append("fkadsTypeId", values.fkadsTypeId);
    formData.formData = 1,
    formData.append(
      "adsStartDate",
      values.adsStartDate ? values.adsStartDate.toISOString() : ""
    );
    formData.append(
      "adsEndDate",
      values.adsEndDate ? values.adsEndDate.toISOString() : ""
    );
    formData.append("lastChanged", new Date().toISOString());
    formData.append("fksellerId", sellerId); 

    if (fileList.length > 0 && fileList[0].originFileObj) {
      formData.append("adsPhoto", fileList[0].originFileObj);
    }

    try {
      if (adsId) {
        await axios.patch(`${URL}/ads-managements/${adsId}`, formData, {
          headers: { "Content-Type": "multipart/form-data" },
        });
        toast.success("Ad updated successfully!");
      } else {
        await axios.post(`${URL}/ads-managements`, formData, {
          headers: { "Content-Type": "multipart/form-data" },
        });
        toast.success("Ad created successfully!");
      }
      navigate("/dashboard/ads-management");
    } catch (error) {
      console.error(error);
      toast.error("Failed to save ad!");
    }
  };

  if (loading) {
    return <Loading fullscreen size={70} />;
  }

  return (
    <>
      {/* Header Section */}
      <Box
        sx={{
          px: { xs: 1.5, sm: 1 },
          pt: 1.2,
          // bgcolor: "#f0f2f5",
          minHeight: "10vh",
        }}
      >
        <Paper
          elevation={4}
          sx={{ p: { xs: 2, sm: 3 }, borderRadius: 4, background: "white" }}
        >
          <Box
            sx={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
              flexWrap: "wrap",
              gap: 2,
            }}
          >
            {/* <Button type="link" icon={<ArrowBackIcon />} style={{ padding: 0 }} onClick={() => navigate(-1)}>
              Back
            </Button> */}
            <Typography
              variant="h6"
              fontWeight={700}
              sx={{
                color: "#1f2937",
                fontSize: { xs: "1.1rem", sm: "1.45rem" },
              }}
            >
              {adsId ? "Update Ads" : "New Ad Form"}
            </Typography>
          </Box>
        </Paper>
      </Box>

      {/* Form Section */}
      <Box sx={{ px: { xs: 1.5, sm: 1 }, py: 1.3 }}>
        <Paper
          elevation={3}
          sx={{ p: { xs: 2, sm: 4 }, borderRadius: 3, background: "white" }}
        >
          <Row justify="center">
            {/* Form Column */}
            <Col xs={24} sm={24} md={16} lg={16}>
              <Form form={form} layout="vertical" onFinish={onFinish}>
                <Form.Item
                  name="adsTitle"
                  label="Ad Title"
                  rules={[{ required: true, message: "Please enter ad title" }]}
                >
                  <Input style={{ width: "70%" }} />
                </Form.Item>

                <Form.Item
                  name="fkadsTypeId"
                  label="Publish Type"
                  rules={[
                    { required: true, message: "Please select publish type" },
                  ]}
                  style={{ width: "70%" }}
                >
                  <Select placeholder="Select publish type">
                    {publishTypes.map((type) => (
                      <Select.Option
                        key={type.adsTypeId}
                        value={type.adsTypeId}
                      >
                        {type.adstypeName}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                <Form.Item name="adsStartDate" label="Ads Start Date">
                  <DatePicker showTime style={{ width: "70%" }} />
                </Form.Item>

                <Form.Item name="adsEndDate" label="Ads End Date">
                  <DatePicker showTime style={{ width: "70%" }} />
                </Form.Item>

                <Form.Item
                  name="adsPhoto"
                  label="Upload Image"
                  style={{ width: "70%" }}
                >
                  <Upload
                    listType="picture"
                    beforeUpload={() => false}
                    fileList={fileList}
                    onChange={({ fileList: newFileList }) =>
                      setFileList(newFileList)
                    }
                    multiple={false}
                    maxCount={1}
                     accept="image/*"
                  >
                    <Button icon={<UploadOutlined />}>Click to Upload</Button>
                  </Upload>
                </Form.Item>

                <Form.Item>
                  <Button
                    type="primary"
                    htmlType="submit"
                    icon={<CheckCircleOutlined />}
                    style={{
                      padding: "10px 24px",
                      minWidth: "150px",
                      fontWeight: 600,
                      borderRadius: "2px",
                      background: "linear-gradient(90deg, #28a745, #34d058)",
                      border: "none",
                      boxShadow: "0 4px 10px rgba(40, 167, 69, 0.3)",
                    }}
                  >
                    {adsId ? "Update Ad" : "Create Ad"}
                  </Button>
                </Form.Item>
              </Form>
            </Col>

            {/* Side Image Column */}
            <Col
              xs={24}
              sm={24}
              md={8}
              lg={8}
              style={{
                background: "rgb(245, 245, 245)",
                display: "flex",
                justifyContent: "center",
                alignItems: "center",
                padding: "10px",
              }}
            >
              <img
                src={Ads}
                alt="Ad Banner"
                style={{
                  maxWidth: "100%",
                  height: "auto",
                  borderRadius: "8px",
                }}
              />
            </Col>
          </Row>
        </Paper>
      </Box>
    </>
  );
};

export default AdsForm;
