import React, { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { Rate, Modal, Tag, message as antMessage } from "antd";
import {
  HeartOutlined,
  ExclamationCircleOutlined,
  CrownOutlined,
} from "@ant-design/icons";
import axios from "axios";
import img1 from "../../assets/PostIMG/PostImg_1.jpeg";
import img2 from "../../assets/PostIMG/PostImg_10.jfif";
import { URL } from "../../config";
import PhoneModal from "../../Components/Modal/PhoneModal";
import OTPModal from "../../Components/Modal/OtpVerificationmodal";
import MembershipPromptModal from "../../Components/Modal/MembershipPromptModal/MembershipPromptModal";
import PackageLimitModal from "../../Components/Modal/PackageLimitModal/PackageLimitModal"; // ✅ नवीन इम्पोर्ट

// Helper: Format price
const formatIndianPrice = (price) => {
  if (!price) return "N/A";
  const numPrice = parseInt(price);
  if (numPrice >= 10000000) return `₹ ${(numPrice / 10000000).toFixed(2)} Cr`;
  if (numPrice >= 100000) return `₹ ${(numPrice / 100000).toFixed(2)} Lakh`;
  if (numPrice >= 1000) return `₹ ${(numPrice / 1000).toFixed(2)} K`;
  return `₹ ${numPrice.toLocaleString("en-IN")}`;
};

const RecommendedProperties = ({ properties = [] }) => {
  const navigate = useNavigate();
  const [favorites, setFavorites] = useState([]);
  const [visibleCards, setVisibleCards] = useState([]);
  const [user, setUser] = useState(null);
  const [activePackage, setActivePackage] = useState(null);
  const [viewCount, setViewCount] = useState(0);
  const [isCheckingLimit, setIsCheckingLimit] = useState(false);

  // ✅ PhoneModal States
  const [phoneModalOpen, setPhoneModalOpen] = useState(false);
  const [regData, setRegData] = useState({ contactno: "" });
  const [pendingPropertyId, setPendingPropertyId] = useState(null);

  // ✅ OTPModal States
  const [otpModalOpen, setOtpModalOpen] = useState(false);
  const [phoneNumber, setPhoneNumber] = useState("");

  // ✅ नवीन MembershipPromptModal स्टेट
  const [membershipModalOpen, setMembershipModalOpen] = useState(false);

  // ✅ नवीन PackageLimitModal स्टेट
  const [packageLimitModalOpen, setPackageLimitModalOpen] = useState(false);

  // Get user from localStorage
  useEffect(() => {
    const storedUser = localStorage.getItem("user");
    if (storedUser) {
      const parsedUser = JSON.parse(storedUser);
      setUser(parsedUser);

      // Check if user has active package
      if (parsedUser.activePackage) {
        const packageEndDate = new Date(parsedUser.activePackage.endDate);
        const currentDate = new Date();

        if (packageEndDate > currentDate) {
          setActivePackage(parsedUser.activePackage);
        }
      }
    }
  }, []);

  // Fetch current view count
  useEffect(() => {
    if (user) {
      fetchViewCount();
    }
  }, [user]);

  // Function to fetch user's current view count
  const fetchViewCount = async () => {
    try {
      let url = "";

      if (user?.fkusertypeId === 1 && user?.buyerId) {
        url = `${URL}/log-properties/buyer/${user.buyerId}`;
      } else if (user?.fkusertypeId !== 1 && user?.sellerId) {
        url = `${URL}/log-properties/seller/${user.sellerId}`;
      }

      if (url) {
        const response = await axios.get(url);
        setViewCount(response.data.length);
      }
    } catch (error) {
      console.error("Error fetching view count:", error);
    }
  };

  // Function to check if user can view more properties
  const canViewProperty = () => {
    // If no package, allow viewing (or you can restrict this based on your business logic)
    if (!activePackage) {
      return true; // You can change this to false if you want to restrict views without package
    }

    const viewLimit = activePackage.propertyVisit || 0;

    // If unlimited views (0 or negative limit), allow
    if (viewLimit <= 0) {
      return true;
    }

    // Check if user has exceeded limit
    return viewCount < viewLimit;
  };

  // Function to show package upgrade modal
  const showUpgradeModal = () => {
    const viewLimit = activePackage?.propertyVisit || 0;

    Modal.confirm({
      title: (
        <div style={{ display: "flex", alignItems: "center", gap: 8 }}>
          <ExclamationCircleOutlined
            style={{ color: "#ff4d4f", fontSize: 24 }}
          />
          <span>View Limit Reached</span>
        </div>
      ),
      icon: null,
      content: (
        <div style={{ padding: "16px 0" }}>
          {activePackage ? (
            <>
              <p style={{ fontSize: 16, marginBottom: 12 }}>
                You've reached your property view limit of{" "}
                <strong>{viewLimit} views</strong>
                for your <strong>{activePackage.packageName}</strong> package.
              </p>
              <p style={{ fontSize: 14, color: "#666", marginBottom: 8 }}>
                To view more properties, please upgrade your package or wait for
                your package to renew.
              </p>
            </>
          ) : (
            <>
              <p style={{ fontSize: 16, marginBottom: 12 }}>
                You don't have an active package.
              </p>
              <p style={{ fontSize: 14, color: "#666", marginBottom: 8 }}>
                Purchase a package to view properties and access premium
                features.
              </p>
            </>
          )}

          <div
            style={{
              marginTop: 16,
              padding: 16,
              background: "#f0f5ff",
              borderRadius: 8,
              border: "1px solid #d6e4ff",
            }}
          >
            <div
              style={{
                display: "flex",
                alignItems: "center",
                gap: 8,
                marginBottom: 8,
              }}
            >
              <CrownOutlined style={{ color: "#1890ff", fontSize: 20 }} />
              <strong style={{ color: "#1890ff" }}>
                Upgrade to Lello Prime
              </strong>
            </div>
            <ul style={{ margin: 0, paddingLeft: 20, color: "#595959" }}>
              <li>Unlimited property views</li>
              <li>Save unlimited properties</li>
              <li>Direct contact with owners</li>
              <li>Priority customer support</li>
            </ul>
          </div>
        </div>
      ),
      okText: "Upgrade Package",
      cancelText: "Cancel",
      okButtonProps: {
        style: {
          backgroundColor: "#1890ff",
          borderColor: "#1890ff",
        },
      },
      width: 500,
      onOk: () => {
        navigate("/membership");
      },
    });
  };

  // Intersection Observer for scroll-based animation
  useEffect(() => {
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            const cards = entry.target.querySelectorAll(".property-card");
            cards.forEach((card, index) => {
              setTimeout(() => {
                card.classList.add("card-visible");
              }, index * 150);
            });
            observer.unobserve(entry.target);
          }
        });
      },
      { threshold: 0.1 },
    );

    const section = document.querySelector(".properties-section");
    if (section) {
      observer.observe(section);
    }

    return () => {
      if (section) {
        observer.unobserve(section);
      }
    };
  }, [properties]);

  // Function to log property view
  const logPropertyView = async (propertyId) => {
    try {
      console.log("=== Starting Property View Log ===");
      console.log("Property ID:", propertyId);

      // Get user data from localStorage
      const storedUser = localStorage.getItem("user");
      const userTypeId = localStorage.getItem("userTypeId");
      const sellerId = localStorage.getItem("sellerId");
      const buyerId = localStorage.getItem("buyerId");
      const role = localStorage.getItem("role");

      console.log("Stored User:", storedUser);
      console.log("User Type ID:", userTypeId);
      console.log("Seller ID:", sellerId);
      console.log("Buyer ID:", buyerId);
      console.log("Role:", role);

      let finalUserId = null;
      let finalUserTypeId = null;
      let userObj = null;

      // Try to parse user object if it exists
      if (storedUser) {
        try {
          userObj = JSON.parse(storedUser);
          console.log("Parsed User Object:", userObj);
        } catch (e) {
          console.log("Error parsing user:", e);
        }
      }

      // ✅ येथे बदल केला आहे - आता आपण fkuserId वापरू
      // Determine user type and ID based on role
      if (role === "seller" || userTypeId === "3") {
        // Seller user type
        finalUserTypeId = userObj?.fkuserId || 3; // ✅ बदल: fkuserId वापरा
        finalUserId = sellerId || (userObj && userObj.sellerId);
        console.log("User Type: Seller");
      } else if (role === "buyer" || userTypeId === "1") {
        // Buyer user type
        finalUserTypeId = userObj?.fkuserId || 1; // ✅ बदल: fkuserId वापरा
        finalUserId =
          buyerId ||
          (userObj && (userObj.buyerId || userObj.userId || userObj.id));
        console.log("User Type: Buyer");
      } else {
        // Try to determine from stored data
        if (sellerId) {
          finalUserTypeId = userObj?.fkuserId || 3; // ✅ बदल: fkuserId वापरा
          finalUserId = sellerId;
        } else if (buyerId) {
          finalUserTypeId = userObj?.fkuserId || 1; // ✅ बदल: fkuserId वापरा
          finalUserId = buyerId;
        } else if (userObj) {
          // Last attempt from user object
          if (userObj.sellerId) {
            finalUserTypeId = userObj?.fkuserId || 3; // ✅ बदल: fkuserId वापरा
            finalUserId = userObj.sellerId;
          } else if (userObj.buyerId) {
            finalUserTypeId = userObj?.fkuserId || 1; // ✅ बदल: fkuserId वापरा
            finalUserId = userObj.buyerId;
          }
        }
      }

      console.log("Final User ID:", finalUserId);
      console.log("Final User Type ID:", finalUserTypeId);

      // Check if we have required data
      if (!finalUserId || !finalUserTypeId) {
        console.log("❌ No user ID or user type found, skipping log");
        return;
      }

      // Get current date and time
      const now = new Date();
      const isoDateTime = now.toISOString();

      // Set time to start of day for the date field (keeping it as ISO format)
      const dateAtMidnight = new Date(now);
      dateAtMidnight.setHours(0, 0, 0, 0);
      const dateISO = dateAtMidnight.toISOString();

      // Format time as HH:MM:SS
      const time = now.toTimeString().split(" ")[0];

      console.log("Date (ISO):", dateISO);
      console.log("Time:", time);
      console.log("ISO DateTime:", isoDateTime);

      // Prepare log data according to LogProperty model
      const logData = {
        fkBuyerIdSellerId: parseInt(finalUserId),
        fkPostpropertyId: parseInt(propertyId),
        fkUsertypeId: parseInt(finalUserTypeId), // ✅ आता हे fkuserId वापरेल
        date: dateISO,
        time: time,
        isDefault: 0,
        isOn: 1,
        isActive: 1,
        createdDate: isoDateTime,
        createdBy: parseInt(finalUserId),
        lastChanged: isoDateTime,
      };

      console.log("Sending Log Data:", logData);
      console.log("API URL:", `${URL}/log-properties`);

      // Post to log-properties endpoint
      const response = await axios.post(`${URL}/log-properties`, logData);

      console.log("✅ Property view logged successfully");
      console.log("Response:", response.data);
      console.log("=== End Property View Log ===");

      // Update local view count
      setViewCount((prev) => prev + 1);
    } catch (error) {
      console.error("❌ Error logging property view:");
      console.error("Error message:", error.message);
      console.error("Error response:", error.response?.data);
      console.error("Error status:", error.response?.status);
      console.log("=== End Property View Log (ERROR) ===");
      // Don't block navigation if logging fails
    }
  };

  // ✅ Updated handleCardClick to open PhoneModal for non-logged-in users
  const handleCardClick = async (propertyId, e) => {
    // Prevent default if event is provided
    if (e) {
      e.preventDefault();
    }

    console.log("Card clicked, Property ID:", propertyId);

    // ✅ Check if user is logged in - if not, open PhoneModal
    if (!user) {
      setPendingPropertyId(propertyId); // Store property ID
      setPhoneModalOpen(true); // Open phone modal
      return;
    }

    setIsCheckingLimit(true);

    try {
      // ✅ नवीन तर्क: जर युजरला अॅक्टिव्ह पॅकेज नाही, तर MembershipPromptModal दाखवा
      if (!activePackage) {
        setIsCheckingLimit(false);
        setMembershipModalOpen(true); // ✅ नवीन मोडल उघडा
        return;
      }

      // ✅ नवीन तर्क: जर युजरची पॅकेज मर्यादा संपली आहे, तर PackageLimitModal दाखवा
      if (!canViewProperty()) {
        setIsCheckingLimit(false);
        setPackageLimitModalOpen(true); // ✅ नवीन मोडल उघडा
        return;
      }

      // Log the property view
      await logPropertyView(propertyId);

      // Navigate to property details
      navigate(`/Propertyinfo/${propertyId}`, { state: { id: propertyId } });
    } catch (error) {
      console.error("Error handling property click:", error);
      antMessage.error("Something went wrong. Please try again.");
    } finally {
      setIsCheckingLimit(false);
    }
  };

  // ✅ Handle PhoneModal close
  const handlePhoneModalClose = () => {
    setPhoneModalOpen(false);
    setPendingPropertyId(null);
    setRegData({ contactno: "" });
  };

  // ✅ Handle Phone submit - Close PhoneModal and Open OTPModal
  const handlePhoneSubmit = () => {
    setPhoneNumber(regData.contactno); // Store phone number for OTP modal
    setPhoneModalOpen(false); // Close phone modal
    setOtpModalOpen(true); // Open OTP modal
  };

  // ✅ Handle going back to PhoneModal from OTPModal
  const handleEditPhoneClick = () => {
    setOtpModalOpen(false); // Close OTP modal
    setPhoneModalOpen(true); // Reopen phone modal
  };

  // ✅ Handle OTP verification success
  const handleOtpSuccess = (userData) => {
    console.log("OTP Success - User Data:", userData);

    // Store user data
    localStorage.setItem("user", JSON.stringify(userData));
    setUser(userData);

    // Close OTP modal
    setOtpModalOpen(false);

    // Reload user from localStorage (in case it was updated)
    const storedUser = localStorage.getItem("user");
    if (storedUser) {
      const parsedUser = JSON.parse(storedUser);
      setUser(parsedUser);

      // Check if user has active package
      if (parsedUser.activePackage) {
        const packageEndDate = new Date(parsedUser.activePackage.endDate);
        const currentDate = new Date();

        if (packageEndDate > currentDate) {
          setActivePackage(parsedUser.activePackage);
        }
      }
    }

    // ✅ OTP व्हेरिफिकेशननंतर, जर युजरला अॅक्टिव्ह पॅकेज नाही, तर MembershipPromptModal दाखवा
    if (!userData.activePackage) {
      setMembershipModalOpen(true);
      return;
    }

    // Navigate to the pending property if there is one
    if (pendingPropertyId) {
      navigate(`/Propertyinfo/${pendingPropertyId}`, {
        state: { id: pendingPropertyId },
      });
      setPendingPropertyId(null);
    }
  };

  // ✅ Handle OTPModal close
  const handleOtpModalClose = () => {
    setOtpModalOpen(false);
    setPhoneNumber("");
    setRegData({ contactno: "" });
    setPendingPropertyId(null);
  };

  // ✅ MembershipPromptModal हँडलर
  const handleMembershipModalClose = () => {
    setMembershipModalOpen(false);
  };

  // ✅ PackageLimitModal हँडलर
  const handlePackageLimitModalClose = () => {
    setPackageLimitModalOpen(false);
  };

  const toggleFavorite = (id) => {
    setFavorites((prev) =>
      prev.includes(id) ? prev.filter((fav) => fav !== id) : [...prev, id],
    );
  };

  const handleViewMore = () => navigate("/properties");

  return (
    <>
      <div
        className="properties-section"
        style={{
          position: "relative",
          padding: "60px 20px",
          backgroundColor: "#f8fbff",
          fontFamily:
            '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif',
        }}
      >
        {/* Background Gradient */}
        <div
          style={{
            position: "absolute",
            top: "60%",
            left: 0,
            right: 0,
            height: "45%",
            background: "linear-gradient(180deg, #e3f2fd 0%, #f0f7ff 100%)",
            zIndex: 0,
          }}
        />

        <div
          style={{
            maxWidth: "1200px",
            margin: "0 auto",
            position: "relative",
            zIndex: 1,
          }}
        >
          {/* Header Section with Animation */}
          <div
            className="header-animate"
            style={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
              marginBottom: "50px",
              flexWrap: "wrap",
              opacity: 0,
              animation: "fadeInDown 0.6s ease forwards",
            }}
          >
            <div>
              <h2
                style={{
                  fontSize: "clamp(26px, 5vw, 42px)",
                  fontWeight: 700,
                  margin: "0 0 8px 0",
                  background: "linear-gradient(135deg, #1976d2, #42a5f5)",
                  WebkitBackgroundClip: "text",
                  WebkitTextFillColor: "transparent",
                  fontFamily: "Poppins",
                }}
              >
                Recommended Properties
              </h2>
              <p
                style={{
                  fontSize: "16px",
                  color: "#666",
                  margin: 0,
                  fontFamily: "Poppins",
                }}
              >
                Based on your recent search
              </p>
            </div>

            <button
              onClick={handleViewMore}
              className="view-more-btn"
              style={{
                backgroundColor: "#1976d2",
                color: "white",
                border: "none",
                padding: "12px 28px",
                borderRadius: "8px",
                fontSize: "15px",
                fontWeight: 600,
                cursor: "pointer",
                transition: "all 0.3s ease",
                boxShadow: "0 2px 8px rgba(25,118,210,0.3)",
                display: "flex",
                alignItems: "center",
                gap: "8px",
                fontFamily: "Poppins",
              }}
            >
              View More ➜
            </button>
          </div>

          {/* Property Cards */}
          {properties.length > 0 ? (
            <div
              className="properties-grid"
              style={{
                display: "grid",
                gridTemplateColumns: "repeat(2, 1fr)",
                gap: "30px",
                padding: "0 10px",
              }}
            >
              {properties.map((property, index) => {
                const imageSrc =
                  property.photo && property.photo.trim() !== ""
                    ? property.photo
                    : property.photos && property.photos.length > 0
                      ? property.photos[0]
                      : index % 2 === 0
                        ? img1
                        : img2;

                const isFav = favorites.includes(property.postpropertyId);
                const isVisible = visibleCards.includes(index);

                const bedrooms = property.noofbedrooms || 0;
                const bathrooms = property.noofbathrooms || 0;
                const area =
                  property.carpetarea ||
                  property.builduparea ||
                  property.supperbuilduparea ||
                  0;
                const propertyTypeName =
                  property.propertyType?.propertytypeName || "Property";
                const mainPropertyType =
                  property.mainPropertyType?.mainprpoertyName || "";
                const postType = property.postType?.posttypeName || "";
                const sellerName = property.seller?.sellerName || "";
                const cityName = property.fkcityId || "";
                const localityName = property.fklocalityId || "";

                return (
                  <div
                    key={property.postpropertyId || index}
                    className={`property-card ${isVisible ? "card-visible" : ""}`}
                    style={{
                      backgroundColor: "white",
                      borderRadius: "16px",
                      overflow: "hidden",
                      boxShadow: "0 4px 14px rgba(0,0,0,0.08)",
                      cursor: isCheckingLimit ? "wait" : "pointer",
                      transition: "transform 0.3s ease, box-shadow 0.3s ease",
                      display: "flex",
                      flexDirection: "row",
                      height: "220px",
                      opacity: isCheckingLimit ? 0.7 : 1,
                      transform: "translateY(30px)",
                    }}
                  >
                    {/* Image Section */}
                    <div
                      className="property-image-container"
                      style={{
                        width: "200px",
                        minWidth: "200px",
                        backgroundColor: "#f0f0f0",
                        overflow: "hidden",
                        position: "relative",
                      }}
                      onClick={(e) =>
                        handleCardClick(property.postpropertyId || index, e)
                      }
                    >
                      <img
                        src={imageSrc}
                        alt={property.Property_Title || "Property"}
                        style={{
                          width: "100%",
                          height: "100%",
                          objectFit: "cover",
                          transition: "transform 0.4s ease",
                        }}
                      />

                      {/* Verified Badge */}
                      <div
                        style={{
                          position: "absolute",
                          top: "10px",
                          right: "10px",
                          backgroundColor: "#4caf50",
                          color: "white",
                          padding: "4px 10px",
                          borderRadius: "20px",
                          fontSize: "11px",
                          fontWeight: 600,
                        }}
                      >
                        ✓ Verified
                      </div>

                      {/* View Limit Badge - Show remaining views if package exists */}
                      {activePackage && activePackage.propertyVisit > 0 && (
                        <Tag
                          style={{
                            position: "absolute",
                            top: "10px",
                            left: "10px",
                            backgroundColor:
                              viewCount >= activePackage.propertyVisit
                                ? "#ff4d4f"
                                : viewCount >= activePackage.propertyVisit * 0.8
                                  ? "#faad14"
                                  : "#1890ff",
                            color: "white",
                            border: "none",
                            fontWeight: 600,
                            padding: "2px 8px",
                            borderRadius: 4,
                            fontSize: "11px",
                          }}
                        >
                          {activePackage.propertyVisit - viewCount} left
                        </Tag>
                      )}
                    </div>

                    {/* Content Section */}
                    <div
                      style={{
                        flex: 1,
                        padding: "20px",
                        display: "flex",
                        flexDirection: "column",
                        justifyContent: "space-between",
                      }}
                    >
                      <div>
                        <h3
                          style={{
                            fontSize: "18px",
                            fontWeight: 600,
                            color: "#1a1a1a",
                            marginBottom: "6px",
                            fontFamily: "Poppins",
                            display: "-webkit-box",
                            WebkitLineClamp: 1,
                            WebkitBoxOrient: "vertical",
                            overflow: "hidden",
                          }}
                        >
                          {property.Property_Title || "Beautiful Home"}
                        </h3>

                        <p
                          style={{
                            fontSize: "14px",
                            color: "#666",
                            marginBottom: "8px",
                            fontFamily: "Poppins",
                          }}
                        >
                          📍{" "}
                          {localityName && localityName !== "hh"
                            ? `${localityName}, `
                            : ""}
                          {cityName || "Prime Location"}
                        </p>

                        {/* Property Details */}
                        <div
                          style={{
                            display: "flex",
                            gap: "12px",
                            marginBottom: "8px",
                            fontSize: "13px",
                            color: "#888",
                            fontFamily: "Poppins",
                          }}
                        >
                          {bedrooms > 0 && <span>🛏️ {bedrooms} BHK</span>}
                          {bathrooms > 0 && <span>🚿 {bathrooms} Bath</span>}
                          {area > 0 && <span>📐 {area} sqft</span>}
                        </div>

                        {/* Property Type Badges */}
                        {mainPropertyType && (
                          <div
                            style={{
                              display: "inline-block",
                              backgroundColor: "#e3f2fd",
                              color: "#1976d2",
                              padding: "4px 12px",
                              borderRadius: "12px",
                              fontSize: "12px",
                              fontWeight: 600,
                              marginBottom: "6px",
                              fontFamily: "Poppins",
                            }}
                          >
                            {mainPropertyType}
                          </div>
                        )}

                        {propertyTypeName && (
                          <div
                            style={{
                              display: "inline-block",
                              backgroundColor: "#e3f2fd",
                              color: "#1976d2",
                              padding: "4px 12px",
                              borderRadius: "12px",
                              fontSize: "12px",
                              fontWeight: 600,
                              marginBottom: "6px",
                              fontFamily: "Poppins",
                              marginLeft: "4px",
                            }}
                          >
                            {propertyTypeName}
                          </div>
                        )}
                      </div>

                      <div
                        style={{
                          display: "flex",
                          justifyContent: "space-between",
                          alignItems: "center",
                          flexWrap: "wrap",
                          gap: "10px",
                        }}
                      >
                        <div>
                          <p
                            style={{
                              fontSize: "12px",
                              color: "#999",
                              marginBottom: "4px",
                              fontFamily: "Poppins",
                            }}
                          >
                            {postType === "Sell" ? "Price" : "Starting from"}
                          </p>
                          <p
                            style={{
                              fontSize: "22px",
                              fontWeight: 700,
                              color: "#1976d2",
                              margin: 0,
                              fontFamily: "Poppins",
                            }}
                          >
                            {formatIndianPrice(property.price)}
                          </p>
                        </div>
                        <button
                          onClick={(e) =>
                            handleCardClick(property.postpropertyId || index, e)
                          }
                          className="details-btn"
                          style={{
                            backgroundColor: "#1976d2",
                            color: "white",
                            border: "none",
                            padding: "10px 18px",
                            borderRadius: "8px",
                            fontSize: "14px",
                            fontWeight: 600,
                            cursor: isCheckingLimit ? "wait" : "pointer",
                            fontFamily: "Poppins",
                            transition: "all 0.3s ease",
                          }}
                        >
                          View Details
                        </button>
                      </div>
                    </div>
                  </div>
                );
              })}
            </div>
          ) : (
            <div
              className="no-properties"
              style={{
                textAlign: "center",
                padding: "60px 20px",
                backgroundColor: "white",
                borderRadius: "16px",
                boxShadow: "0 2px 8px rgba(0,0,0,0.05)",
                opacity: 0,
                animation: "fadeIn 0.6s ease forwards 0.3s",
              }}
            >
              <p style={{ fontSize: "18px", color: "#666" }}>
                No recommended properties found.
              </p>
            </div>
          )}
        </div>

        {/* Enhanced Styles with Animations */}
        <style>{`
    @keyframes fadeInDown {
      from {
        opacity: 0;
        transform: translateY(-20px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    @keyframes fadeIn {
      from {
        opacity: 0;
      }
      to {
        opacity: 1;
      }
    }

    @keyframes slideInUp {
      from {
        opacity: 0;
        transform: translateY(30px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    .property-card.card-visible {
      animation: slideInUp 0.5s ease forwards;
    }

    .property-card:hover {
      transform: translateY(-8px) !important;
      box-shadow: 0 12px 24px rgba(0,0,0,0.15) !important;
    }

    .property-card:hover .property-image-container img {
      transform: scale(1.1);
    }

    .view-more-btn:hover {
      background-color: #1565c0 !important;
      transform: translateY(-2px);
      box-shadow: 0 4px 12px rgba(25,118,210,0.4) !important;
    }

    .details-btn:hover {
      background-color: #1565c0 !important;
      transform: translateX(4px);
    }

    .favorite-icon:hover {
      transform: scale(1.15) !important;
    }

    .favorite-icon:active {
      transform: scale(0.95) !important;
    }

    @media (max-width: 1100px) {
      .properties-grid {
        grid-template-columns: 1fr !important;
      }
    }

    @media (max-width: 768px) {
      .property-card {
        flex-direction: column !important;
        height: auto !important;
      }

      .property-image-container {
        width: 100% !important;
        min-width: unset !important;
        height: 220px !important;
      }

      .property-card img {
        width: 100% !important;
        height: 100% !important;
        border-radius: 16px 16px 0 0 !important;
      }

      .property-card > div:last-child {
        padding: 16px !important;
      }

      .details-btn {
        padding: 8px 16px !important;
        font-size: 13px !important;
      }
    }

    @media (max-width: 480px) {
      .property-image-container {
        height: 200px !important;
      }

      .property-card h3 {
        font-size: 16px !important;
      }

      .property-card p {
        font-size: 13px !important;
      }
    }
  `}</style>
      </div>

      {/* ✅ PhoneModal Component */}
      <PhoneModal
        open={phoneModalOpen}
        handleClose={handlePhoneModalClose}
        regData={regData}
        setRegData={setRegData}
        onSubmit={handlePhoneSubmit}
      />

      {/* ✅ OTPModal Component */}
      <OTPModal
        open={otpModalOpen}
        onClose={handleOtpModalClose}
        phone={phoneNumber}
        onEditPhoneClick={handleEditPhoneClick}
        handleOtpSuccess={handleOtpSuccess}
      />

      {/* ✅ MembershipPromptModal Component */}
      <MembershipPromptModal
        open={membershipModalOpen}
        onClose={handleMembershipModalClose}
      />

      {/* ✅ PackageLimitModal Component */}
      <PackageLimitModal
        open={packageLimitModalOpen}
        onClose={handlePackageLimitModalClose}
        packageName={activePackage?.packageName}
      />
    </>
  );
};

export default RecommendedProperties;