import React, { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { motion } from "framer-motion";
import axios from "axios";
import {
  Box,
  Grid,
  Card,
  CardMedia,
  CardContent,
  Typography,
  Button,
  Chip,
  Stack,
  Rating,
  Container,
  Avatar,
  IconButton,
  Tooltip,
} from "@mui/material";
import {
  LocationOn,
  Bed,
  Bathtub,
  SquareFoot,
  Balcony,
  Apartment,
  Person,
  FavoriteBorder,
  Share,
  Verified,
  Elevator,
  LocalParking,
  Security,
} from "@mui/icons-material";
import { Modal, Tag, message as antMessage } from "antd";
import { ExclamationCircleOutlined, CrownOutlined } from "@ant-design/icons";
import img1 from "../../assets/PostIMG/PostImg_3.jpeg";
import img2 from "../../assets/PostIMG/PostImg_4.jpeg";
import { URL } from "../../config";
import PhoneModal from "../../Components/Modal/PhoneModal";
import OTPModal from "../../Components/Modal/OtpVerificationmodal";
import MembershipPromptModal from "../../Components/Modal/MembershipPromptModal/MembershipPromptModal"; // ✅ नवीन इम्पोर्ट

const fallbackImages = [img1, img2];

// Function to format price in Lakh and CR format
const formatPrice = (price) => {
  if (!price) return "N/A";

  const crore = 10000000; // 1 CR = 1,00,00,000
  const lakh = 100000; // 1 Lakh = 1,00,000

  if (price >= crore) {
    return `${(price / crore).toFixed(2)} CR`;
  } else if (price >= lakh) {
    return `${(price / lakh).toFixed(2)} Lakh`;
  } else {
    return `₹${price.toLocaleString()}`;
  }
};

const PropertyListings = ({
  properties = [],
  title = "Lello Certified Properties",
}) => {
  const navigate = useNavigate();
  const [user, setUser] = useState(null);
  const [activePackage, setActivePackage] = useState(null);
  const [viewCount, setViewCount] = useState(0);
  const [isCheckingLimit, setIsCheckingLimit] = useState(false);

  // ✅ PhoneModal States
  const [phoneModalOpen, setPhoneModalOpen] = useState(false);
  const [regData, setRegData] = useState({ contactno: "" });
  const [pendingPropertyId, setPendingPropertyId] = useState(null);

  // ✅ OTPModal States
  const [otpModalOpen, setOtpModalOpen] = useState(false);
  const [phoneNumber, setPhoneNumber] = useState("");

  // ✅ नवीन MembershipPromptModal स्टेट
  const [membershipModalOpen, setMembershipModalOpen] = useState(false);

  // Get user from localStorage
  useEffect(() => {
    const storedUser = localStorage.getItem("user");
    if (storedUser) {
      const parsedUser = JSON.parse(storedUser);
      setUser(parsedUser);

      // Check if user has active package
      if (parsedUser.activePackage) {
        const packageEndDate = new Date(parsedUser.activePackage.endDate);
        const currentDate = new Date();

        if (packageEndDate > currentDate) {
          setActivePackage(parsedUser.activePackage);
        }
      }
    }
  }, []);

  // Fetch current view count
  useEffect(() => {
    if (user) {
      fetchViewCount();
    }
  }, [user]);

  // Function to fetch user's current view count
  const fetchViewCount = async () => {
    try {
      let url = "";

      if (user?.fkusertypeId === 1 && user?.buyerId) {
        url = `${URL}/log-properties/buyer/${user.buyerId}`;
      } else if (user?.fkusertypeId !== 1 && user?.sellerId) {
        url = `${URL}/log-properties/seller/${user.sellerId}`;
      }

      if (url) {
        const response = await axios.get(url);
        setViewCount(response.data.length);
      }
    } catch (error) {
      console.error("Error fetching view count:", error);
    }
  };

  // Function to check if user can view more properties
  const canViewProperty = () => {
    // If no package, allow viewing (or you can restrict this based on your business logic)
    if (!activePackage) {
      return true; // You can change this to false if you want to restrict views without package
    }

    const viewLimit = activePackage.propertyVisit || 0;

    // If unlimited views (0 or negative limit), allow
    if (viewLimit <= 0) {
      return true;
    }

    // Check if user has exceeded limit
    return viewCount < viewLimit;
  };

  // Function to show package upgrade modal
  const showUpgradeModal = () => {
    const viewLimit = activePackage?.propertyVisit || 0;

    Modal.confirm({
      title: (
        <div style={{ display: "flex", alignItems: "center", gap: 8 }}>
          <ExclamationCircleOutlined
            style={{ color: "#ff4d4f", fontSize: 24 }}
          />
          <span>View Limit Reached</span>
        </div>
      ),
      icon: null,
      content: (
        <div style={{ padding: "16px 0" }}>
          {activePackage ? (
            <>
              <p style={{ fontSize: 16, marginBottom: 12 }}>
                You've reached your property view limit of{" "}
                <strong>{viewLimit} views</strong>
                for your <strong>{activePackage.packageName}</strong> package.
              </p>
              <p style={{ fontSize: 14, color: "#666", marginBottom: 8 }}>
                To view more properties, please upgrade your package or wait for
                your package to renew.
              </p>
            </>
          ) : (
            <>
              <p style={{ fontSize: 16, marginBottom: 12 }}>
                You don't have an active package.
              </p>
              <p style={{ fontSize: 14, color: "#666", marginBottom: 8 }}>
                Purchase a package to view properties and access premium
                features.
              </p>
            </>
          )}

          <div
            style={{
              marginTop: 16,
              padding: 16,
              background: "#f0f5ff",
              borderRadius: 8,
              border: "1px solid #d6e4ff",
            }}
          >
            <div
              style={{
                display: "flex",
                alignItems: "center",
                gap: 8,
                marginBottom: 8,
              }}
            >
              <CrownOutlined style={{ color: "#1890ff", fontSize: 20 }} />
              <strong style={{ color: "#1890ff" }}>
                Upgrade to Lello Prime
              </strong>
            </div>
            <ul style={{ margin: 0, paddingLeft: 20, color: "#595959" }}>
              <li>Unlimited property views</li>
              <li>Save unlimited properties</li>
              <li>Direct contact with owners</li>
              <li>Priority customer support</li>
            </ul>
          </div>
        </div>
      ),
      okText: "Upgrade Package",
      cancelText: "Cancel",
      okButtonProps: {
        style: {
          backgroundColor: "#1890ff",
          borderColor: "#1890ff",
        },
      },
      width: 500,
      onOk: () => {
        navigate("/membership");
      },
    });
  };

  // Function to log property view - Updated to use /log-properties endpoint
  const logPropertyView = async (propertyId) => {
    try {
      console.log("=== Starting Property View Log (PropertyListings) ===");
      console.log("Property ID:", propertyId);

      // Get user data from localStorage
      const storedUser = localStorage.getItem("user");
      const userTypeId = localStorage.getItem("userTypeId");
      const sellerId = localStorage.getItem("sellerId");
      const buyerId = localStorage.getItem("buyerId");
      const role = localStorage.getItem("role");

      console.log("Stored User:", storedUser);
      console.log("User Type ID:", userTypeId);
      console.log("Seller ID:", sellerId);
      console.log("Buyer ID:", buyerId);
      console.log("Role:", role);

      let finalUserId = null;
      let finalUserTypeId = null;
      let userObj = null;

      // Try to parse user object if it exists
      if (storedUser) {
        try {
          userObj = JSON.parse(storedUser);
          console.log("Parsed User Object:", userObj);
        } catch (e) {
          console.log("Error parsing user:", e);
        }
      }

      // ✅ येथे बदल केला आहे - आता आपण fkuserId वापरू
      // Determine user type and ID based on role
      if (role === "seller" || userTypeId === "3") {
        // Seller user type
        finalUserTypeId = userObj?.fkuserId || 3; // ✅ बदल: fkuserId वापरा
        finalUserId = sellerId || (userObj && userObj.sellerId);
        console.log("User Type: Seller");
      } else if (role === "buyer" || userTypeId === "1") {
        // Buyer user type
        finalUserTypeId = userObj?.fkuserId || 1; // ✅ बदल: fkuserId वापरा
        finalUserId =
          buyerId ||
          (userObj && (userObj.buyerId || userObj.userId || userObj.id));
        console.log("User Type: Buyer");
      } else {
        // Try to determine from stored data
        if (sellerId) {
          finalUserTypeId = userObj?.fkuserId || 3; // ✅ बदल: fkuserId वापरा
          finalUserId = sellerId;
        } else if (buyerId) {
          finalUserTypeId = userObj?.fkuserId || 1; // ✅ बदल: fkuserId वापरा
          finalUserId = buyerId;
        } else if (userObj) {
          // Last attempt from user object
          if (userObj.sellerId) {
            finalUserTypeId = userObj?.fkuserId || 3; // ✅ बदल: fkuserId वापरा
            finalUserId = userObj.sellerId;
          } else if (userObj.buyerId) {
            finalUserTypeId = userObj?.fkuserId || 1; // ✅ बदल: fkuserId वापरा
            finalUserId = userObj.buyerId;
          }
        }
      }

      console.log("Final User ID:", finalUserId);
      console.log("Final User Type ID:", finalUserTypeId);

      // Check if we have required data
      if (!finalUserId || !finalUserTypeId) {
        console.log("❌ No user ID or user type found, skipping log");
        return;
      }

      // Get current date and time
      const now = new Date();
      const isoDateTime = now.toISOString();

      // Set time to start of day for the date field (keeping it as ISO format)
      const dateAtMidnight = new Date(now);
      dateAtMidnight.setHours(0, 0, 0, 0);
      const dateISO = dateAtMidnight.toISOString();

      // Format time as HH:MM:SS
      const time = now.toTimeString().split(" ")[0];

      console.log("Date (ISO):", dateISO);
      console.log("Time:", time);
      console.log("ISO DateTime:", isoDateTime);

      // Prepare log data according to LogProperty model (same as RecommendedProperties)
      const logData = {
        fkBuyerIdSellerId: parseInt(finalUserId),
        fkPostpropertyId: parseInt(propertyId),
        fkUsertypeId: parseInt(finalUserTypeId), // ✅ आता हे fkuserId वापरेल
        date: dateISO,
        time: time,
        isDefault: 0,
        isOn: 1,
        isActive: 1,
        createdDate: isoDateTime,
        createdBy: parseInt(finalUserId),
        lastChanged: isoDateTime,
      };

      console.log("Sending Log Data:", logData);
      console.log("API URL:", `${URL}/log-properties`);

      // Post to log-properties endpoint (changed from log-buyers)
      const response = await axios.post(`${URL}/log-properties`, logData);

      console.log("✅ Property view logged successfully");
      console.log("Response:", response.data);
      console.log("=== End Property View Log ===");

      // Update local view count
      setViewCount((prev) => prev + 1);
    } catch (error) {
      console.error("❌ Error logging property view:");
      console.error("Error message:", error.message);
      console.error("Error response:", error.response?.data);
      console.error("Error status:", error.response?.status);
      console.log("=== End Property View Log (ERROR) ===");
      // Don't block navigation if logging fails
    }
  };

  // ✅ Updated handleCardClick to open PhoneModal for non-logged-in users
  const handleCardClick = async (propertyId) => {
    console.log("Card clicked, Property ID:", propertyId);

    // ✅ Check if user is logged in - if not, open PhoneModal
    if (!user) {
      setPendingPropertyId(propertyId); // Store property ID
      setPhoneModalOpen(true); // Open phone modal
      return;
    }

    setIsCheckingLimit(true);

    try {
      // ✅ नवीन तर्क: जर युजरला अॅक्टिव्ह पॅकेज नाही, तर MembershipPromptModal दाखवा
      if (!activePackage) {
        setIsCheckingLimit(false);
        setMembershipModalOpen(true); // ✅ नवीन मोडल उघडा
        return;
      }

      // Check if user can view property
      if (!canViewProperty()) {
        setIsCheckingLimit(false);
        showUpgradeModal();
        return;
      }

      // Log the property view
      await logPropertyView(propertyId);

      // Store in localStorage and navigate
      localStorage.setItem("id", propertyId);
      navigate(`/Propertyinfo/${propertyId}`);
    } catch (error) {
      console.error("Error handling property click:", error);
      antMessage.error("Something went wrong. Please try again.");
    } finally {
      setIsCheckingLimit(false);
    }
  };

  // ✅ Updated handleViewDetails to open PhoneModal for non-logged-in users
  const handleViewDetails = async (propertyId, e) => {
    // Prevent event propagation
    e.stopPropagation();

    console.log("View Details clicked, Property ID:", propertyId);

    // ✅ Check if user is logged in - if not, open PhoneModal
    if (!user) {
      setPendingPropertyId(propertyId); // Store property ID
      setPhoneModalOpen(true); // Open phone modal
      return;
    }

    setIsCheckingLimit(true);

    try {
      // ✅ नवीन तर्क: जर युजरला अॅक्टिव्ह पॅकेज नाही, तर MembershipPromptModal दाखवा
      if (!activePackage) {
        setIsCheckingLimit(false);
        setMembershipModalOpen(true); // ✅ नवीन मोडल उघडा
        return;
      }

      // Check if user can view property
      if (!canViewProperty()) {
        setIsCheckingLimit(false);
        showUpgradeModal();
        return;
      }

      // Log the property view
      await logPropertyView(propertyId);

      // Store in localStorage and navigate
      localStorage.setItem("id", propertyId);
      navigate(`/Propertyinfo/${propertyId}`);
    } catch (error) {
      console.error("Error handling property click:", error);
      antMessage.error("Something went wrong. Please try again.");
    } finally {
      setIsCheckingLimit(false);
    }
  };

  // ✅ Handle PhoneModal close
  const handlePhoneModalClose = () => {
    setPhoneModalOpen(false);
    setPendingPropertyId(null);
    setRegData({ contactno: "" });
  };

  // ✅ Handle Phone submit - Close PhoneModal and Open OTPModal
  const handlePhoneSubmit = () => {
    setPhoneNumber(regData.contactno); // Store phone number for OTP modal
    setPhoneModalOpen(false); // Close phone modal
    setOtpModalOpen(true); // Open OTP modal
  };

  // ✅ Handle going back to PhoneModal from OTPModal
  const handleEditPhoneClick = () => {
    setOtpModalOpen(false); // Close OTP modal
    setPhoneModalOpen(true); // Reopen phone modal
  };

  // ✅ Handle OTP verification success
  const handleOtpSuccess = (userData) => {
    console.log("OTP Success - User Data:", userData);

    // Store user data
    localStorage.setItem("user", JSON.stringify(userData));
    setUser(userData);

    // Close OTP modal
    setOtpModalOpen(false);

    // Reload user from localStorage (in case it was updated)
    const storedUser = localStorage.getItem("user");
    if (storedUser) {
      const parsedUser = JSON.parse(storedUser);
      setUser(parsedUser);

      // Check if user has active package
      if (parsedUser.activePackage) {
        const packageEndDate = new Date(parsedUser.activePackage.endDate);
        const currentDate = new Date();

        if (packageEndDate > currentDate) {
          setActivePackage(parsedUser.activePackage);
        }
      }
    }

    // ✅ OTP व्हेरिफिकेशननंतर, जर युजरला अॅक्टिव्ह पॅकेज नाही, तर MembershipPromptModal दाखवा
    if (!parsedUser.activePackage) {
      setMembershipModalOpen(true);
      return;
    }

    // Navigate to the pending property if there is one
    if (pendingPropertyId) {
      navigate(`/Propertyinfo/${pendingPropertyId}`, {
        state: { id: pendingPropertyId },
      });
      setPendingPropertyId(null);
    }
  };

  // ✅ Handle OTPModal close
  const handleOtpModalClose = () => {
    setOtpModalOpen(false);
    setPhoneNumber("");
    setRegData({ contactno: "" });
    setPendingPropertyId(null);
  };

  // ✅ MembershipPromptModal हँडलर
  const handleMembershipModalClose = () => {
    setMembershipModalOpen(false);
  };

  return (
    <>
      <div style={{ backgroundColor: "#f8f9fa" }}>
        <Container
          maxWidth="lg"
          sx={{
            py: { xs: 5, md: 8 },
          }}
        >
          {/* Header Section */}
          <Box
            sx={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
              mb: 6,
              flexWrap: "wrap",
              gap: 2,
            }}
          >
            <Box>
              <Typography
                variant="h2"
                sx={{
                  fontSize: { xs: "28px", md: "42px" },
                  fontWeight: 700,
                  mb: 1,
                  background: "linear-gradient(135deg, #1976d2, #42a5f5)",
                  WebkitBackgroundClip: "text",
                  WebkitTextFillColor: "transparent",
                  fontFamily: "Poppins",
                }}
              >
                Lello Certified Properties
              </Typography>
              <Typography
                variant="body1"
                sx={{
                  fontSize: "16px",
                  color: "#666",
                  fontFamily: "Poppins",
                }}
              >
                Discover unbeatable deals and exceptional value on your next
                property investment.
              </Typography>
            </Box>

            <Button
              variant="contained"
              sx={{
                backgroundColor: "#1976d2",
                color: "white",
                px: 4,
                py: 1.5,
                borderRadius: "8px",
                fontSize: "15px",
                fontWeight: 600,
                boxShadow: "0 4px 12px rgba(25,118,210,0.3)",
                transition: "all 0.3s ease",
                "&:hover": {
                  backgroundColor: "#1565c0",
                  transform: "translateY(-2px)",
                  boxShadow: "0 6px 20px rgba(25,118,210,0.4)",
                },
                fontFamily: "Poppins",
              }}
            >
              View More ➜
            </Button>
          </Box>

          {/* Property Cards Grid */}
          <Grid container spacing={4} sx={{ display: "flex", justifyContent: "space-between", }}>
            {properties.length > 0 ? (
              properties.map((property, index) => {
                const imageSrc =
                  property.photo && property.photo.trim() !== ""
                    ? property.photo
                    : fallbackImages[index % fallbackImages.length];

                const rating = property.rating || 4.2;

                return (
                  <Grid
                    item
                    xs={12}
                    sm={6}
                    lg={4}
                    component={motion.div}
                    key={property.postpropertyId || index}
                    initial={{ opacity: 0, y: 50 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    viewport={{ once: false, amount: 0.3 }}
                    transition={{ duration: 0.6, delay: index * 0.15 }}
                  >
                    <Card
                      onClick={() => handleCardClick(property.postpropertyId)}
                      sx={{
                        borderRadius: "16px",
                        overflow: "hidden",
                        cursor: isCheckingLimit ? "wait" : "pointer",
                        transition: "all 0.3s ease",
                        height: "100%",
                        display: "flex",
                        flexDirection: "column",
                        boxShadow: "0 8px 24px rgba(0,0,0,0.1)",
                        opacity: isCheckingLimit ? 0.7 : 1,
                        "&:hover": {
                          transform: isCheckingLimit
                            ? "none"
                            : "translateY(-8px)",
                          boxShadow: "0 12px 32px rgba(0,0,0,0.15)",
                        },
                      }}
                    >
                      {/* Image Section with Overlay */}
                      <Box sx={{ position: "relative" }}>
                        <CardMedia
                          component="img"
                          image={imageSrc}
                          alt={property.Property_Title || "Property"}
                          onError={(e) => {
                            e.target.onerror = null;
                            e.target.src =
                              fallbackImages[index % fallbackImages.length];
                          }}
                          sx={{
                            height: { xs: 220, sm: 240, md: 260 },
                            width: { xs: 220, sm: 240, md: 335 },
                            objectFit: "cover",
                          }}
                        />

                        {/* Gradient Overlay */}
                        <Box
                          sx={{
                            position: "absolute",
                            top: 0,
                            left: 0,
                            right: 0,
                            bottom: 0,
                            background:
                              "linear-gradient(to bottom, transparent 60%, rgba(0,0,0,0.7) 100%)",
                          }}
                        />

                        {/* Top Badges */}
                        <Stack
                          direction="row"
                          spacing={1}
                          sx={{
                            position: "absolute",
                            top: 12,
                            left: 12,
                            zIndex: 1,
                          }}
                        >
                          <Chip
                            icon={<Verified sx={{ fontSize: 14 }} />}
                            label="Certified"
                            size="small"
                            sx={{
                              bgcolor: "rgba(255,255,255,0.95)",
                              color: "#1976d2",
                              fontWeight: 600,
                              fontSize: "0.7rem",
                              borderRadius: "20px",
                              backdropFilter: "blur(10px)",
                            }}
                          />
                        </Stack>

                        <Stack
                          direction="row"
                          spacing={1}
                          sx={{
                            position: "absolute",
                            top: 12,
                            right: 12,
                            zIndex: 1,
                          }}
                        >
                          <Chip
                            label={
                              property.postType?.posttypeName || "For Sale"
                            }
                            size="small"
                            sx={{
                              bgcolor: "rgba(255, 152, 0, 0.9)",
                              color: "#fff",
                              fontWeight: "bold",
                              fontSize: "0.7rem",
                              borderRadius: "20px",
                              backdropFilter: "blur(10px)",
                            }}
                          />
                        </Stack>

                        {/* View Limit Badge - Show remaining views if package exists */}
                        {activePackage && activePackage.propertyVisit > 0 && (
                          <Chip
                            label={`${activePackage.propertyVisit - viewCount} views left`}
                            size="small"
                            sx={{
                              position: "absolute",
                              bottom: 12,
                              right: 12,
                              bgcolor:
                                viewCount >= activePackage.propertyVisit
                                  ? "rgba(255, 77, 79, 0.9)"
                                  : viewCount >=
                                      activePackage.propertyVisit * 0.8
                                    ? "rgba(250, 173, 20, 0.9)"
                                    : "rgba(24, 144, 255, 0.9)",
                              color: "#fff",
                              fontWeight: "bold",
                              fontSize: "0.7rem",
                              borderRadius: "20px",
                              backdropFilter: "blur(10px)",
                            }}
                          />
                        )}
                      </Box>

                      {/* Content Section */}
                      <CardContent
                        sx={{
                          p: 3,
                          flexGrow: 1,
                          display: "flex",
                          flexDirection: "column",
                          bgcolor: "white",
                        }}
                      >
                        {/* Title and Location */}
                        <Box mb={2}>
                          <Typography
                            variant="h6"
                            sx={{
                              fontSize: { xs: "1.1rem", md: "1.25rem" },
                              fontWeight: 700,
                              color: "#1a1a1a",
                              mb: 0.5,
                              lineHeight: 1.3,
                              fontFamily: "Poppins",
                            }}
                          >
                            {property.Property_Title || "Beautiful Home"}
                          </Typography>

                          <Stack
                            direction="row"
                            alignItems="center"
                            spacing={0.5}
                            sx={{ mt: 0.5 }}
                          >
                            <LocationOn
                              sx={{
                                fontSize: 16,
                                color: "#1976d2",
                              }}
                            />
                            <Typography
                              variant="body2"
                              sx={{
                                color: "#666",
                                fontSize: "0.9rem",
                                fontFamily: "Poppins",
                              }}
                            >
                              {property.fkcityId ||
                                property.fklocalityId ||
                                "Location not available"}
                            </Typography>
                          </Stack>
                        </Box>

                        {/* Property Type Badge */}
                        <Stack
                          direction="row"
                          alignItems="center"
                          spacing={1}
                          mb={2}
                        >
                          <Chip
                            icon={<Apartment sx={{ fontSize: 14 }} />}
                            label={`${
                              property.propertyType?.propertytypeName ||
                              "Property"
                            } • ${
                              property.mainPropertyType?.mainprpoertyName ||
                              "Type"
                            }`}
                            size="small"
                            variant="outlined"
                            sx={{
                              borderColor: "#e0e0e0",
                              color: "#666",
                              fontSize: "0.75rem",
                              fontFamily: "Poppins",
                            }}
                          />
                        </Stack>

                        {/* Rating */}
                        <Stack
                          direction="row"
                          alignItems="center"
                          spacing={1}
                          mb={2}
                        >
                          <Rating
                            value={rating}
                            precision={0.5}
                            readOnly
                            size="small"
                            sx={{ color: "#ffc107" }}
                          />
                          <Typography
                            variant="body2"
                            sx={{
                              fontWeight: 600,
                              color: "#333",
                              fontSize: "0.9rem",
                            }}
                          >
                            {rating.toFixed(1)} (24 reviews)
                          </Typography>
                        </Stack>

                        {/* Property Features */}
                        <Box
                          sx={{
                            display: "grid",
                            gridTemplateColumns: "repeat(2, 1fr)",
                            gap: 1.5,
                            mb: 2,
                          }}
                        >
                          {property.noofbedrooms && (
                            <Stack
                              direction="row"
                              alignItems="center"
                              spacing={1}
                            >
                              <Box
                                sx={{
                                  p: 0.5,
                                  borderRadius: 1,
                                  bgcolor: "#f5f5f5",
                                  display: "flex",
                                  alignItems: "center",
                                  justifyContent: "center",
                                }}
                              >
                                <Bed sx={{ fontSize: 18, color: "#1976d2" }} />
                              </Box>
                              <Typography
                                variant="body2"
                                sx={{
                                  color: "#555",
                                  fontSize: "0.85rem",
                                  fontFamily: "Poppins",
                                }}
                              >
                                {property.noofbedrooms} Beds
                              </Typography>
                            </Stack>
                          )}

                          {property.noofbathrooms && (
                            <Stack
                              direction="row"
                              alignItems="center"
                              spacing={1}
                            >
                              <Box
                                sx={{
                                  p: 0.5,
                                  borderRadius: 1,
                                  bgcolor: "#f5f5f5",
                                  display: "flex",
                                  alignItems: "center",
                                  justifyContent: "center",
                                }}
                              >
                                <Bathtub
                                  sx={{ fontSize: 18, color: "#1976d2" }}
                                />
                              </Box>
                              <Typography
                                variant="body2"
                                sx={{
                                  color: "#555",
                                  fontSize: "0.85rem",
                                  fontFamily: "Poppins",
                                }}
                              >
                                {property.noofbathrooms} Baths
                              </Typography>
                            </Stack>
                          )}

                          {property.builduparea && (
                            <Stack
                              direction="row"
                              alignItems="center"
                              spacing={1}
                            >
                              <Box
                                sx={{
                                  p: 0.5,
                                  borderRadius: 1,
                                  bgcolor: "#f5f5f5",
                                  display: "flex",
                                  alignItems: "center",
                                  justifyContent: "center",
                                }}
                              >
                                <SquareFoot
                                  sx={{ fontSize: 18, color: "#1976d2" }}
                                />
                              </Box>
                              <Typography
                                variant="body2"
                                sx={{
                                  color: "#555",
                                  fontSize: "0.85rem",
                                  fontFamily: "Poppins",
                                }}
                              >
                                {property.supperbuilduparea} sqft
                              </Typography>
                            </Stack>
                          )}

                          {property.noofbalkanies && (
                            <Stack
                              direction="row"
                              alignItems="center"
                              spacing={1}
                            >
                              <Box
                                sx={{
                                  p: 0.5,
                                  borderRadius: 1,
                                  bgcolor: "#f5f5f5",
                                  display: "flex",
                                  alignItems: "center",
                                  justifyContent: "center",
                                }}
                              >
                                <Balcony
                                  sx={{ fontSize: 18, color: "#1976d2" }}
                                />
                              </Box>
                              <Typography
                                variant="body2"
                                sx={{
                                  color: "#555",
                                  fontSize: "0.85rem",
                                  fontFamily: "Poppins",
                                }}
                              >
                                {property.noofbalkanies} Balcony
                              </Typography>
                            </Stack>
                          )}
                        </Box>

                        {/* Price Section */}
                        <Box
                          sx={{
                            mt: "auto",
                            pt: 2,
                            borderTop: "1px solid #f0f0f0",
                          }}
                        >
                          <Stack
                            direction="row"
                            justifyContent="space-between"
                            alignItems="center"
                            mb={1.5}
                          >
                            <Box>
                              <Typography
                                variant="body2"
                                sx={{
                                  color: "#666",
                                  fontSize: "0.85rem",
                                  fontFamily: "Poppins",
                                }}
                              >
                                Total Price
                              </Typography>
                              <Typography
                                variant="h5"
                                sx={{
                                  fontWeight: 700,
                                  color: "#1976d2",
                                  fontSize: { xs: "1.3rem", md: "1.5rem" },
                                  fontFamily: "Poppins",
                                }}
                              >
                                {formatPrice(property.price)}
                              </Typography>
                            </Box>

                            <Stack
                              direction="row"
                              alignItems="center"
                              spacing={1}
                            >
                              <Avatar
                                sx={{
                                  width: 32,
                                  height: 32,
                                  bgcolor: "#1976d2",
                                  fontSize: "0.8rem",
                                }}
                              >
                                {property.seller?.sellerName?.charAt(0) || "O"}
                              </Avatar>
                              <Box>
                                <Typography
                                  variant="body2"
                                  sx={{
                                    fontWeight: 600,
                                    color: "#333",
                                    fontSize: "0.85rem",
                                    fontFamily: "Poppins",
                                  }}
                                >
                                  {property.seller?.sellerName || "Owner"}
                                </Typography>
                                <Typography
                                  variant="caption"
                                  sx={{
                                    color: "#999",
                                    fontSize: "0.7rem",
                                    fontFamily: "Poppins",
                                  }}
                                >
                                  Verified Seller
                                </Typography>
                              </Box>
                            </Stack>
                          </Stack>

                          {/* View Details Button */}
                          <Box
                            sx={{ display: "flex", justifyContent: "center" }}
                          >
                            <button
                              onClick={(e) =>
                                handleViewDetails(property.postpropertyId, e)
                              }
                              className="details-btn"
                              style={{
                                backgroundColor: "#1976d2",
                                color: "white",
                                border: "none",
                                padding: "10px 18px",
                                borderRadius: "8px",
                                fontSize: "14px",
                                fontWeight: 600,
                                cursor: isCheckingLimit ? "wait" : "pointer",
                                fontFamily: "Poppins",
                                transition: "all 0.3s ease",
                                width: "100%",
                              }}
                            >
                              View Details
                            </button>
                          </Box>
                        </Box>
                      </CardContent>
                    </Card>
                  </Grid>
                );
              })
            ) : (
              <Box
                sx={{
                  width: "100%",
                  py: 8,
                  textAlign: "center",
                }}
              >
                <Typography
                  variant="h6"
                  sx={{
                    color: "#666",
                    fontFamily: "Poppins",
                  }}
                >
                  No properties found at the moment.
                </Typography>
                <Typography
                  variant="body2"
                  sx={{
                    color: "#999",
                    mt: 1,
                    fontFamily: "Poppins",
                  }}
                >
                  Please check back later for new listings.
                </Typography>
              </Box>
            )}
          </Grid>
        </Container>
      </div>

      {/* ✅ PhoneModal Component */}
      <PhoneModal
        open={phoneModalOpen}
        handleClose={handlePhoneModalClose}
        regData={regData}
        setRegData={setRegData}
        onSubmit={handlePhoneSubmit}
      />

      {/* ✅ OTPModal Component */}
      <OTPModal
        open={otpModalOpen}
        onClose={handleOtpModalClose}
        phone={phoneNumber}
        onEditPhoneClick={handleEditPhoneClick}
        handleOtpSuccess={handleOtpSuccess}
      />

      {/* ✅ MembershipPromptModal Component */}
      <MembershipPromptModal
        open={membershipModalOpen}
        onClose={handleMembershipModalClose}
      />
    </>
  );
};

export default PropertyListings;