import React, { useEffect, useState } from "react";
import {
  Card,
  Checkbox,
  Slider,
  Select,
  Typography,
  Divider,
  Tag,
  Input,
  Rate,
  Row,
  Col,
  Switch,
  Space,
  Button,
  Collapse,
  Radio,
  Tooltip,
} from "antd";
import {
  CheckCircleOutlined,
  HomeOutlined,
  CarOutlined,
  RestOutlined,
  DollarOutlined,
  FilterOutlined,
  ClearOutlined,
  ApartmentOutlined,
  UserOutlined,
  SettingOutlined,
  SearchOutlined,
  StarOutlined,
  PlusOutlined,
  SkinOutlined,
  ToolOutlined,
  ExperimentOutlined,
} from "@ant-design/icons";

const { Title, Text } = Typography;
const { Option } = Select;
const { Panel } = Collapse;

const SidebarFilters = ({
  onFilterChange,
  propertyTypes = [],
  mainProperties = [],
  postTypes = [],
  users = [],
  activeFilters = {},
}) => {
  // Local state synchronized with parent's activeFilters
  const [selectedPostType, setSelectedPostType] = useState(
    activeFilters.postTypeId || null
  );
  const [selectedPropertyType, setSelectedPropertyType] = useState(
    activeFilters.propertyTypeId || null
  );
  const [selectedMainProperty, setSelectedMainProperty] = useState(
    activeFilters.mainPropertyTypeId || null
  );
  const [selectedUserTypeId, setSelectedUserTypeId] = useState(
    activeFilters.userId || null
  );
  const [priceRange, setPriceRange] = useState(
    activeFilters.priceRange || [0, 100000000]
  );

  // New filter states
  const [bedrooms, setBedrooms] = useState(activeFilters.bedrooms || null);
  const [bathrooms, setBathrooms] = useState(activeFilters.bathrooms || null);
  const [areaRange, setAreaRange] = useState(
    activeFilters.areaRange || [0, 5000]
  );
  const [furnishing, setFurnishing] = useState(
    activeFilters.furnishing || null
  );
  const [possession, setPossession] = useState(
    activeFilters.possession || null
  );
  const [parking, setParking] = useState(activeFilters.parking || null);
  const [verified, setVerified] = useState(activeFilters.verified || false);

  const [showMoreBHK, setShowMoreBHK] = useState(false);
  const [selectedPropertyTypes, setSelectedPropertyTypes] = useState(
  activeFilters.propertyTypeIds || []
);

  // Sync local state when activeFilters change from parent
  useEffect(() => {
    if (activeFilters.postTypeId !== undefined) {
      setSelectedPostType(activeFilters.postTypeId);
    }
    if (activeFilters.propertyTypeId !== undefined) {
      setSelectedPropertyType(activeFilters.propertyTypeId);
    }
    if (activeFilters.mainPropertyTypeId !== undefined) {
      setSelectedMainProperty(activeFilters.mainPropertyTypeId);
    }
    if (activeFilters.userId !== undefined) {
      setSelectedUserTypeId(activeFilters.userId);
    }
    if (activeFilters.priceRange) {
      setPriceRange(activeFilters.priceRange);
    }
    if (activeFilters.bedrooms !== undefined) {
      setBedrooms(activeFilters.bedrooms);
    }
    if (activeFilters.bathrooms !== undefined) {
      setBathrooms(activeFilters.bathrooms);
    }
    if (activeFilters.areaRange) {
      setAreaRange(activeFilters.areaRange);
    }
    if (activeFilters.furnishing !== undefined) {
      setFurnishing(activeFilters.furnishing);
    }
    if (activeFilters.possession !== undefined) {
      setPossession(activeFilters.possession);
    }
    if (activeFilters.parking !== undefined) {
      setParking(activeFilters.parking);
    }
    if (activeFilters.verified !== undefined) {
      setVerified(activeFilters.verified);
    }
     if (activeFilters.propertyTypeIds !== undefined) {
    setSelectedPropertyTypes(activeFilters.propertyTypeIds);
  }
  }, [activeFilters]);

  // Send filter changes to parent
  useEffect(() => {
    onFilterChange({
      postTypeId: selectedPostType,
      propertyTypeIds: selectedPropertyTypes,
      mainPropertyTypeId: selectedMainProperty,
      userId: selectedUserTypeId,
      priceRange: priceRange,
      bedrooms: bedrooms,
      bathrooms: bathrooms,
      areaRange: areaRange,
      furnishing: furnishing,
      possession: possession,
      parking: parking,
      verified: verified,
    });
  }, [
    selectedPostType,
    selectedPropertyTypes,
    selectedMainProperty,
    selectedUserTypeId,
    priceRange,
    bedrooms,
    bathrooms,
    areaRange,
    furnishing,
    possession,
    parking,
    verified,
  ]);

  // Format price range for display
  const formatPrice = (value) => {
    if (value >= 10000000) return `₹${(value / 10000000).toFixed(1)}Cr`;
    if (value >= 100000) return `₹${(value / 100000).toFixed(1)}L`;
    return `₹${value}`;
  };

  // Format area for display
  const formatArea = (value) => {
    return `${value} sq.ft.`;
  };

  const clearFiltersRef = React.useRef(() => {});

  // Clear all filters
const clearAllFilters = () => {
  const clearedFilters = {
    postTypeId: null,
    propertyTypeIds: [],
    mainPropertyTypeId: null,
    userId: null,
    priceRange: [0, 100000000],
    bedrooms: null,
    bathrooms: null,
    areaRange: [0, 5000],
    furnishing: null,
    possession: null,
    parking: null,
    verified: false,
  };

  // 🔹 Local state reset
  setSelectedPostType(null);
  setSelectedPropertyTypes([]);
  setSelectedMainProperty(null);
  setSelectedUserTypeId(null);
  setPriceRange([0, 100000000]);
  setBedrooms(null);
  setBathrooms(null);
  setAreaRange([0, 5000]);
  setFurnishing(null);
  setPossession(null);
  setParking(null);
  setVerified(false);

  // 🔹 IMPORTANT: parent ला direct notify
  onFilterChange(clearedFilters);
};


  useEffect(() => {
    clearFiltersRef.current = clearAllFilters;
  }, []);

  // Professional filter button component
  const ProfessionalFilterButton = ({
    icon,
    label,
    active,
    onClick,
    tooltip,
  }) => (
    <Tooltip title={tooltip} placement="top">
      <div
        onClick={onClick}
        style={{
          display: "flex",
          flexDirection: "column",
          alignItems: "center",
          justifyContent: "center",
          width: "90px",
          height: "90px",
          borderRadius: "16px",
          border: active ? "2px solid #2563eb" : "1px solid #e5e7eb",
          backgroundColor: active ? "#eff6ff" : "#ffffff",
          cursor: "pointer",
          transition: "all 0.3s cubic-bezier(0.4, 0, 0.2, 1)",
          boxShadow: active
            ? "0 10px 25px rgba(37, 99, 235, 0.15), 0 4px 12px rgba(0, 0, 0, 0.04)"
            : "0 1px 3px rgba(0, 0, 0, 0.05), 0 1px 2px rgba(0, 0, 0, 0.1)",
          position: "relative",
          overflow: "hidden",
        }}
        onMouseEnter={(e) => {
          if (!active) {
            e.currentTarget.style.transform = "translateY(-4px) scale(1.02)";
            e.currentTarget.style.boxShadow =
              "0 20px 40px rgba(0, 0, 0, 0.1), 0 8px 16px rgba(0, 0, 0, 0.06)";
            e.currentTarget.style.borderColor = "#d1d5db";
          }
        }}
        onMouseLeave={(e) => {
          if (!active) {
            e.currentTarget.style.transform = "translateY(0) scale(1)";
            e.currentTarget.style.boxShadow =
              "0 1px 3px rgba(0, 0, 0, 0.05), 0 1px 2px rgba(0, 0, 0, 0.1)";
            e.currentTarget.style.borderColor = "#e5e7eb";
          }
        }}
      >
        {active && (
          <div
            style={{
              position: "absolute",
              top: "8px",
              right: "8px",
              width: "8px",
              height: "8px",
              borderRadius: "50%",
              backgroundColor: "#2563eb",
            }}
          />
        )}
        <div
          style={{
            fontSize: "28px",
            marginBottom: "8px",
            color: active ? "#2563eb" : "#6b7280",
            transition: "all 0.3s ease",
          }}
        >
          {icon}
        </div>
        <Text
          style={{
            fontSize: "13px",
            fontWeight: active ? 600 : 500,
            color: active ? "#1e40af" : "#374151",
            textAlign: "center",
            fontFamily: "Poppins",
          }}
        >
          {label}
        </Text>
      </div>
    </Tooltip>
  );

  // Professional tag component
  const ProfessionalTag = ({ label, active, onClick, icon }) => (
    <Tag
      onClick={onClick}
      style={{
        cursor: "pointer",
        backgroundColor: active ? "#2563eb" : "#f9fafb",
        color: active ? "#ffffff" : "#374151",
        borderColor: active ? "#2563eb" : "#e5e7eb",
        borderRadius: "8px",
        padding: "6px 14px",
        fontSize: "14px",
        fontWeight: active ? 600 : 500,
        margin: "4px 6px 4px 0",
        transition: "all 0.2s cubic-bezier(0.4, 0, 0.2, 1)",
        fontFamily: "Poppins",
        display: "inline-flex",
        alignItems: "center",
        gap: "6px",
        boxShadow: active
          ? "0 2px 8px rgba(37, 99, 235, 0.3)"
          : "0 1px 2px rgba(0, 0, 0, 0.05)",
      }}
      onMouseEnter={(e) => {
        if (!active) {
          e.currentTarget.style.backgroundColor = "#f3f4f6";
          e.currentTarget.style.borderColor = "#d1d5db";
          e.currentTarget.style.transform = "translateY(-1px)";
          e.currentTarget.style.boxShadow = "0 4px 12px rgba(0, 0, 0, 0.1)";
        }
      }}
      onMouseLeave={(e) => {
        if (!active) {
          e.currentTarget.style.backgroundColor = "#f9fafb";
          e.currentTarget.style.borderColor = "#e5e7eb";
          e.currentTarget.style.transform = "translateY(0)";
          e.currentTarget.style.boxShadow = "0 1px 2px rgba(0, 0, 0, 0.05)";
        }
      }}
    >
      {icon && <span style={{ fontSize: "16px" }}>{icon}</span>}
      {label}
    </Tag>
  );

  return (
    <Card
      bordered={false}
      clearFiltersRef={clearFiltersRef}
      style={{
        borderRadius: "0px",
        boxShadow:
          "0 10px 40px rgba(0, 0, 0, 0.06), 0 2px 10px rgba(0, 0, 0, 0.04)",
        overflow: "hidden",
        fontFamily: "Poppins",
        background: "#ffffff",
      }}
      bodyStyle={{ padding: "0", fontFamily: "Poppins" }}
    >
      {/* Professional Header */}
      <div
        style={{
          padding: "24px 28px",
          borderBottom: "1px solid #f1f5f9",
          background: "linear-gradient(135deg, #667eea 0%, #764ba2 100%)",
          position: "relative",
          overflow: "hidden",
        }}
      >
        <div
          style={{
            position: "absolute",
            top: 0,
            left: 0,
            right: 0,
            bottom: 0,
            background:
              'url(\'data:image/svg+xml,%3Csvg width="60" height="60" viewBox="0 0 60 60" xmlns="http://www.w3.org/2000/svg"%3E%3Cg fill="none" fill-rule="evenodd"%3E%3Cg fill="%23ffffff" fill-opacity="0.05"%3E%3Cpath d="M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z"/%3E%3C/g%3E%3C/g%3E%3C/svg%3E\')',
          }}
        />
        <div style={{ position: "relative", zIndex: 1 }}>
          <div
            style={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
            }}
          >
            <div>
              <Title
                level={3}
                style={{
                  margin: 0,
                  color: "white",
                  fontFamily: "Poppins",
                  fontWeight: 700,
                  fontSize: "20px",
                  letterSpacing: "-0.025em",
                }}
              >
                <FilterOutlined
                  style={{ marginRight: "10px", fontSize: "18px" }}
                />
                Property Filters
              </Title>
              <Text
                style={{
                  color: "rgba(255, 255, 255, 0.8)",
                  fontSize: "13px",
                  marginTop: "4px",
                  display: "block",
                  fontFamily: "Poppins",
                }}
              >
                Refine your property search
              </Text>
            </div>
            <Button
              type="text"
              onClick={clearAllFilters}
              style={{
                color: "white",
                border: "1px solid rgba(255,255,255,0.25)",
                borderRadius: "12px",
                fontFamily: "Poppins",
                fontWeight: 500,
                height: "36px",
                paddingLeft: "12px",
                paddingRight: "12px",
                backgroundColor: "rgba(255, 255, 255, 0.1)",
                backdropFilter: "blur(10px)",
              }}
              icon={<ClearOutlined />}
            >
              Clear All
            </Button>
          </div>
        </div>
      </div>

      <div
        style={{
          padding: "24px 28px",
          maxHeight: "calc(100vh - 170px)",
          overflowY: "auto",
        }}
      >
        {/* Post Type */}
        <div style={{ marginBottom: "28px" }}>
          <div
            style={{
              display: "flex",
              alignItems: "center",
              marginBottom: "16px",
            }}
          >
            <div
              style={{
                width: "4px",
                height: "20px",
                backgroundColor: "#2563eb",
                borderRadius: "2px",
                marginRight: "12px",
              }}
            />
            <Title
              level={5}
              style={{
                margin: 0,
                color: "#1f2937",
                fontFamily: "Poppins",
                fontWeight: 600,
                fontSize: "16px",
              }}
            >
              Property Type
            </Title>
          </div>
          <div style={{ display: "flex", gap: "16px", flexWrap: "wrap" }}>
            {postTypes.length > 0 ? (
              postTypes.map((type) => (
                <ProfessionalFilterButton
                  key={type.postTypeId}
                  icon={
                    type.posttypeName === "Sale"
                      ? "🏷️"
                      : type.posttypeName === "Rent"
                      ? "👥"
                      : "🏠"
                  }
                  label={type.posttypeName}
                  active={selectedPostType === type.postTypeId}
                  onClick={() =>
                    setSelectedPostType((prev) =>
                      prev === type.postTypeId ? null : type.postTypeId
                    )
                  }
                  tooltip={`Filter by ${type.posttypeName} properties`}
                />
              ))
            ) : (
              <Text type="secondary" style={{ fontFamily: "Poppins" }}>
                No post types found
              </Text>
            )}
          </div>
        </div>
        <Divider style={{ margin: "24px 0", borderColor: "#f1f5f9" }} />
        {/* Main Property Type */}
        <div style={{ marginBottom: "28px" }}>
          <div
            style={{
              display: "flex",
              alignItems: "center",
              marginBottom: "16px",
            }}
          >
            <div
              style={{
                width: "4px",
                height: "20px",
                backgroundColor: "#7c3aed",
                borderRadius: "2px",
                marginRight: "12px",
              }}
            />
            <Title
              level={5}
              style={{
                margin: 0,
                color: "#1f2937",
                fontFamily: "Poppins",
                fontWeight: 600,
                fontSize: "16px",
              }}
            >
              Category
            </Title>
          </div>
          <Radio.Group
            value={selectedMainProperty}
            onChange={(e) => setSelectedMainProperty(e.target.value)}
            style={{ width: "100%" }}
          >
            <Space direction="horizontal" style={{ width: "100%" }}>
              {mainProperties.length > 0 ? (
                mainProperties.map((main) => (
                  <Radio
                    key={main.mainpropertytypeId}
                    value={main.mainpropertytypeId}
                    style={{
                      fontFamily: "Poppins",
                      padding: "8px 12px",
                      borderRadius: "8px",
                      margin: "4px 0",
                      border:
                        selectedMainProperty === main.mainpropertytypeId
                          ? "1px solid #2563eb"
                          : "1px solid #e5e7eb",
                      backgroundColor:
                        selectedMainProperty === main.mainpropertytypeId
                          ? "#eff6ff"
                          : "#ffffff",
                    }}
                  >
                    <span
                      style={{
                        display: "flex",
                        alignItems: "center",
                        gap: "10px",
                      }}
                    >
                      <span style={{ fontSize: "16px" }}>
                        {main.mainprpoertyName === "Residential" ? "🏢" : "🏪"}
                      </span>
                      <span style={{ fontWeight: 500 }}>
                        {main.mainprpoertyName}
                      </span>
                    </span>
                  </Radio>
                ))
              ) : (
                <Text type="secondary" style={{ fontFamily: "Poppins" }}>
                  No categories found
                </Text>
              )}
            </Space>
          </Radio.Group>
        </div>
        <Divider style={{ margin: "24px 0", borderColor: "#f1f5f9" }} />
        {/* Budget */}
        <div style={{ marginBottom: "28px" }}>
          <div
            style={{
              display: "flex",
              alignItems: "center",
              marginBottom: "16px",
            }}
          >
            <DollarOutlined
              style={{
                marginRight: "12px",
                color: "#059669",
                fontSize: "18px",
              }}
            />
            <Title
              level={5}
              style={{
                margin: 0,
                color: "#1f2937",
                fontFamily: "Poppins",
                fontWeight: 600,
                fontSize: "16px",
              }}
            >
              Budget Range
            </Title>
          </div>
          <div
            style={{
              padding: "16px 20px",
              backgroundColor: "#f9fafb",
              borderRadius: "12px",
              border: "1px solid #e5e7eb",
            }}
          >
            <Slider
              range
              min={0}
              max={100000000}
              step={100000}
              value={priceRange}
              onChange={(val) => setPriceRange(val)}
              tooltip={{
                formatter: formatPrice,
              }}
              trackStyle={{ backgroundColor: "#059669", height: 6 }}
              handleStyle={{
                borderColor: "#059669",
                backgroundColor: "#ffffff",
                boxShadow: "0 2px 8px rgba(5, 150, 105, 0.3)",
                width: 20,
                height: 20,
              }}
              railStyle={{ backgroundColor: "#e5e7eb", height: 6 }}
            />
            <div
              style={{
                display: "flex",
                justifyContent: "space-between",
                fontSize: "14px",
                color: "#6b7280",
                marginTop: "16px",
                fontFamily: "Poppins",
                fontWeight: 500,
              }}
            >
              <span
                style={{
                  backgroundColor: "#ffffff",
                  padding: "6px 12px",
                  borderRadius: "6px",
                  border: "1px solid #e5e7eb",
                  fontWeight: 600,
                  color: "#059669",
                }}
              >
                {formatPrice(priceRange[0])}
              </span>
              <span
                style={{
                  backgroundColor: "#ffffff",
                  padding: "6px 12px",
                  borderRadius: "6px",
                  border: "1px solid #e5e7eb",
                  fontWeight: 600,
                  color: "#059669",
                }}
              >
                {formatPrice(priceRange[1])}
              </span>
            </div>
          </div>
          <Row gutter={12} style={{ marginTop: "16px" }}>
            <Col span={12}>
              <Select
                placeholder="Min Budget"
                style={{ width: "100%", fontFamily: "Poppins" }}
                value={priceRange[0] || undefined}
                onChange={(val) => setPriceRange([val, priceRange[1]])}
                size="middle"
                bordered={false}
                style={{
                  backgroundColor: "#f9fafb",
                  border: "1px solid #e5e7eb",
                  borderRadius: "8px",
                }}
              >
                <Option style={{ fontFamily: "Poppins" }} value={0}>
                  ₹0
                </Option>
                <Option style={{ fontFamily: "Poppins" }} value={1000000}>
                  ₹10L
                </Option>
                <Option style={{ fontFamily: "Poppins" }} value={2000000}>
                  ₹20L
                </Option>
                <Option style={{ fontFamily: "Poppins" }} value={5000000}>
                  ₹50L
                </Option>
                <Option style={{ fontFamily: "Poppins" }} value={10000000}>
                  ₹1Cr
                </Option>
              </Select>
            </Col>
            <Col span={12}>
              <Select
                placeholder="Max Budget"
                style={{ width: "100%", fontFamily: "Poppins" }}
                value={priceRange[1] === 100000000 ? undefined : priceRange[1]}
                onChange={(val) => setPriceRange([priceRange[0], val])}
                size="middle"
                bordered={false}
                style={{
                  backgroundColor: "#f9fafb",
                  border: "1px solid #e5e7eb",
                  borderRadius: "8px",
                }}
              >
                <Option style={{ fontFamily: "Poppins" }} value={5000000}>
                  ₹50L
                </Option>
                <Option style={{ fontFamily: "Poppins" }} value={10000000}>
                  ₹1Cr
                </Option>
                <Option style={{ fontFamily: "Poppins" }} value={20000000}>
                  ₹2Cr
                </Option>
                <Option style={{ fontFamily: "Poppins" }} value={50000000}>
                  ₹5Cr
                </Option>
                <Option style={{ fontFamily: "Poppins" }} value={100000000}>
                  ₹10Cr+
                </Option>
              </Select>
            </Col>
          </Row>
        </div>
        <Divider style={{ margin: "24px 0", borderColor: "#f1f5f9" }} />
        {/* Bedrooms */}
        <div style={{ marginBottom: "28px" }}>
          <div
            style={{
              display: "flex",
              alignItems: "center",
              marginBottom: "16px",
            }}
          >
            <span
              style={{
                marginRight: "12px",
                color: "#dc2626",
                fontSize: "18px",
              }}
            >
              🛏️
            </span>
            <Title
              level={5}
              style={{
                margin: 0,
                color: "#1f2937",
                fontFamily: "Poppins",
                fontWeight: 600,
                fontSize: "16px",
              }}
            >
              Bedrooms
            </Title>
          </div>
          <div style={{ display: "flex", flexWrap: "wrap", gap: "8px" }}>
            {[1, 2, 3, 4, 5].map((num) => (
              <ProfessionalTag
                key={num}
                label={`${num} ${num === 1 ? "BHK" : "BHK"}`}
                active={bedrooms === num}
                onClick={() => setBedrooms(bedrooms === num ? null : num)}
                icon={<HomeOutlined />}
              />
            ))}

            {/* More button */}
            <ProfessionalTag
              label={showMoreBHK ? "Less" : "More"}
              active={showMoreBHK}
              onClick={() => setShowMoreBHK(!showMoreBHK)}
              icon={<PlusOutlined />}
            />

            {/* 6–9+ BHK */}
            {showMoreBHK &&
              [6, 7, 8, 9, "9+"].map((num) => (
                <ProfessionalTag
                  key={num}
                  label={`${num} BHK`}
                  active={bedrooms === num}
                  onClick={() => setBedrooms(bedrooms === num ? null : num)}
                  icon={<ApartmentOutlined />}
                />
              ))}
          </div>
        </div>
        <Divider style={{ margin: "24px 0", borderColor: "#f1f5f9" }} />
        {/* Area */}
        <div style={{ marginBottom: "28px" }}>
          <div
            style={{
              display: "flex",
              alignItems: "center",
              marginBottom: "16px",
            }}
          >
            <HomeOutlined
              style={{
                marginRight: "12px",
                color: "#7c3aed",
                fontSize: "18px",
              }}
            />
            <Title
              level={5}
              style={{
                margin: 0,
                color: "#1f2937",
                fontFamily: "Poppins",
                fontWeight: 600,
                fontSize: "16px",
              }}
            >
              Area (sq.ft.)
            </Title>
          </div>
          <div
            style={{
              padding: "16px 20px",
              backgroundColor: "#faf5ff",
              borderRadius: "12px",
              border: "1px solid #e9d5ff",
            }}
          >
            <Slider
              range
              min={0}
              max={5000}
              step={100}
              value={areaRange}
              onChange={(val) => setAreaRange(val)}
              tooltip={{
                formatter: formatArea,
              }}
              trackStyle={{ backgroundColor: "#7c3aed", height: 6 }}
              handleStyle={{
                borderColor: "#7c3aed",
                backgroundColor: "#ffffff",
                boxShadow: "0 2px 8px rgba(124, 58, 237, 0.3)",
                width: 20,
                height: 20,
              }}
              railStyle={{ backgroundColor: "#e9d5ff", height: 6 }}
            />
            <div
              style={{
                display: "flex",
                justifyContent: "space-between",
                fontSize: "14px",
                color: "#6b7280",
                marginTop: "16px",
                fontFamily: "Poppins",
                fontWeight: 500,
              }}
            >
              <span
                style={{
                  backgroundColor: "#ffffff",
                  padding: "6px 12px",
                  borderRadius: "6px",
                  border: "1px solid #e9d5ff",
                  fontWeight: 600,
                  color: "#7c3aed",
                }}
              >
                {formatArea(areaRange[0])}
              </span>
              <span
                style={{
                  backgroundColor: "#ffffff",
                  padding: "6px 12px",
                  borderRadius: "6px",
                  border: "1px solid #e9d5ff",
                  fontWeight: 600,
                  color: "#7c3aed",
                }}
              >
                {formatArea(areaRange[1])}
              </span>
            </div>
          </div>
        </div>
        <Divider style={{ margin: "24px 0", borderColor: "#f1f5f9" }} />
        {/* Property Type */}

        {/* Property Types */}
       <div style={{ marginBottom: "28px" }}>
  <div
    style={{
      display: "flex",
      alignItems: "center",
      marginBottom: "16px",
    }}
  >
    <ApartmentOutlined
      style={{
        marginRight: "12px",
        color: "#ea580c",
        fontSize: "18px",
      }}
    />
    <Title
      level={5}
      style={{
        margin: 0,
        color: "#1f2937",
        fontFamily: "Poppins",
        fontWeight: 600,
        fontSize: "16px",
      }}
    >
      Property Types
    </Title>
  </div>
  <div style={{ display: "flex", flexWrap: "wrap", gap: "8px" }}>
    {propertyTypes.length > 0 ? (
      propertyTypes.map((type) => (
        <ProfessionalTag
          key={type.propertyTypeId}
          label={type.propertytypeName}
          active={selectedPropertyTypes.includes(type.propertyTypeId)}
          onClick={() => {
            setSelectedPropertyTypes((prev) => {
              if (prev.includes(type.propertyTypeId)) {
                // Remove if already selected
                return prev.filter((id) => id !== type.propertyTypeId);
              } else {
                // Add if not selected
                return [...prev, type.propertyTypeId];
              }
            });
          }}
          icon={<ApartmentOutlined />}
        />
      ))
    ) : (
      <Text type="secondary" style={{ fontFamily: "Poppins" }}>
        No property types found
      </Text>
    )}
  </div>
</div>
        <Divider style={{ margin: "24px 0", borderColor: "#f1f5f9" }} />
        {/* Posted By */}
        <div style={{ marginBottom: "28px" }}>
          <div
            style={{
              display: "flex",
              alignItems: "center",
              marginBottom: "16px",
            }}
          >
            <UserOutlined
              style={{
                marginRight: "12px",
                color: "#0891b2",
                fontSize: "18px",
              }}
            />
            <Title
              level={5}
              style={{
                margin: 0,
                color: "#1f2937",
                fontFamily: "Poppins",
                fontWeight: 600,
                fontSize: "16px",
              }}
            >
              Posted By
            </Title>
          </div>
          <div style={{ display: "flex", flexWrap: "wrap", gap: "8px" }}>
            {users.length > 0 ? (
              users.map((user) => (
                <ProfessionalTag
                  key={user.userId}
                  label={user.username || user.type || user.role}
                  active={selectedUserTypeId === user.userId}
                  onClick={() =>
                    setSelectedUserTypeId((prev) =>
                      prev === user.userId ? null : user.userId
                    )
                  }
                  icon={
                    <UserOutlined
                      style={{
                        borderRadius: 50,
                        border: "1px solid black",
                        padding: 2,
                      }}
                    />
                  }
                />
              ))
            ) : (
              <Text type="secondary" style={{ fontFamily: "Poppins" }}>
                No user types found
              </Text>
            )}
          </div>
        </div>
        <Divider style={{ margin: "24px 0", borderColor: "#f1f5f9" }} />
        {/* Furnishing */}
        <div style={{ marginBottom: "28px" }}>
          <div
            style={{
              display: "flex",
              alignItems: "center",
              marginBottom: "16px",
            }}
          >
            <SettingOutlined
              style={{
                marginRight: "12px",
                color: "#65a30d",
                fontSize: "18px",
              }}
            />
            <Title
              level={5}
              style={{
                margin: 0,
                color: "#1f2937",
                fontFamily: "Poppins",
                fontWeight: 600,
                fontSize: "16px",
              }}
            >
              Furnishing Status
            </Title>
          </div>
          <div style={{ display: "flex", flexWrap: "wrap", gap: "8px" }}>
            {["Furnished", "Semi-Furnished", "Unfurnished"].map((type) => (
              <ProfessionalTag
                key={type}
                label={type}
                active={furnishing === type.toLowerCase().replace("-", "")}
                onClick={() =>
                  setFurnishing(
                    furnishing === type.toLowerCase().replace("-", "")
                      ? null
                      : type.toLowerCase().replace("-", "")
                  )
                }
                icon={
                  type === "Furnished" ? (
                    <SkinOutlined />
                  ) : type === "Semi-Furnished" ? (
                    <ToolOutlined />
                  ) : (
                    <HomeOutlined />
                  )
                }
              />
            ))}
          </div>
        </div>
        <Divider style={{ margin: "24px 0", borderColor: "#f1f5f9" }} />
        {/* Bathrooms */}
        <div style={{ marginBottom: "28px" }}>
          <div
            style={{
              display: "flex",
              alignItems: "center",
              marginBottom: "16px",
            }}
          >
            <RestOutlined
              style={{
                marginRight: "12px",
                color: "#2563eb",
                fontSize: "18px",
              }}
            />
            <Title
              level={5}
              style={{
                margin: 0,
                color: "#1f2937",
                fontFamily: "Poppins",
                fontWeight: 600,
                fontSize: "16px",
              }}
            >
              Bathrooms
            </Title>
          </div>
          <div style={{ display: "flex", flexWrap: "wrap", gap: "8px" }}>
            {[1, 2, 3, 4, 5].map((num) => (
              <ProfessionalTag
                key={num}
                label={`${num} ${num === 1 ? "Bathroom" : "Bathrooms"}`}
                active={bathrooms === num}
                onClick={() => setBathrooms(bathrooms === num ? null : num)}
                icon={<ExperimentOutlined />}
              />
            ))}
          </div>
        </div>
        <Divider style={{ margin: "24px 0", borderColor: "#f1f5f9" }} />
        {/* Parking */}
        {/* <div style={{ marginBottom: "28px" }}>
          <div
            style={{
              display: "flex",
              alignItems: "center",
              marginBottom: "16px",
            }}
          >
            <CarOutlined
              style={{
                marginRight: "12px",
                color: "#0891b2",
                fontSize: "18px",
              }}
            />
            <Title
              level={5}
              style={{
                margin: 0,
                color: "#1f2937",
                fontFamily: "Poppins",
                fontWeight: 600,
                fontSize: "16px",
              }}
            >
              Parking
            </Title>
          </div>
          <div style={{ display: "flex", flexWrap: "wrap", gap: "8px" }}>
            {["Open", "Covered", "Both"].map((type) => (
              <ProfessionalTag
                key={type}
                label={type}
                active={parking === type.toLowerCase()}
                onClick={() =>
                  setParking(
                    parking === type.toLowerCase() ? null : type.toLowerCase()
                  )
                }
                icon="🚗"
              />
            ))}
          </div>
        </div>

        <Divider style={{ margin: "24px 0", borderColor: "#f1f5f9" }} /> */}
        {/* Verified Properties */}
        <div style={{ marginBottom: "20px" }}>
          <div
            style={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
              padding: "16px 20px",
              backgroundColor: "#f0fdf4",
              borderRadius: "12px",
              border: "1px solid #bbf7d0",
              transition: "all 0.3s ease",
            }}
            onMouseEnter={(e) => {
              e.currentTarget.style.boxShadow =
                "0 4px 12px rgba(34, 197, 94, 0.1)";
              e.currentTarget.style.transform = "translateY(-2px)";
            }}
            onMouseLeave={(e) => {
              e.currentTarget.style.boxShadow = "none";
              e.currentTarget.style.transform = "translateY(0)";
            }}
          >
            <div>
              <Text
                strong
                style={{
                  fontFamily: "Poppins",
                  color: "#166534",
                  fontSize: "15px",
                  display: "flex",
                  alignItems: "center",
                  gap: "8px",
                }}
              >
                <CheckCircleOutlined
                  style={{ color: "#22c55e", marginRight: 4 }}
                />
                Verified Properties Only
              </Text>
              <div
                style={{
                  fontSize: "13px",
                  color: "#16a34a",
                  marginTop: "6px",
                  fontFamily: "Poppins",
                }}
              >
                Show only verified and authenticated properties
              </div>
            </div>
            <Switch
              checked={verified}
              onChange={(checked) => setVerified(checked)}
              style={{ backgroundColor: verified ? "#22c55e" : undefined }}
              size="default"
            />
          </div>
        </div>
      </div>
    </Card>
  );
};

export default SidebarFilters;
