import { useState, useEffect } from "react";
import {
  Row,
  Col,
  Typography,
  Button,
  Image,
  Tag,
  Card,
  Divider,
  message,
  Modal,
  Alert,
  notification,
} from "antd";
import { 
  PhoneOutlined, 
  HeartOutlined, 
  HeartFilled, 
  ExclamationCircleOutlined,
  WarningOutlined,
  CloseCircleOutlined,
} from "@ant-design/icons";
import img2 from "../../assets/images/building2.png";
import axios from "axios";
import { URL } from "../../config";
import { useNavigate } from "react-router-dom";

const { Title, Text } = Typography;

const PropertyPhotos = ({ property }) => {
  const [saved, setSaved] = useState(false);
  const [loading, setLoading] = useState(false);
  const [user, setUser] = useState(null);
  const [activePackage, setActivePackage] = useState(null);
  const [cartCount, setCartCount] = useState(0);
  const [checkingLimit, setCheckingLimit] = useState(false);
  const navigate = useNavigate();

  // Get user from localStorage on component mount
  useEffect(() => {
    const storedUser = localStorage.getItem("user");
    if (storedUser) {
      const parsedUser = JSON.parse(storedUser);
      setUser(parsedUser);
      
      // Check if user has active package
      if (parsedUser.activePackage) {
        const packageEndDate = new Date(parsedUser.activePackage.endDate);
        const currentDate = new Date();
        
        if (packageEndDate > currentDate) {
          setActivePackage(parsedUser.activePackage);
        }
      }
    }
  }, []);

  // Fetch cart count when user is loaded
  useEffect(() => {
    if (user) {
      fetchCartCount();
    }
  }, [user]);

  // Check if property is already saved
  useEffect(() => {
    const checkIfSaved = async () => {
      if (!user || !property) return;

      try {
        let endpoint;
        if (user.buyerId) {
          endpoint = `${URL}/add-to-carts/buyer/${user.buyerId}`;
        } else if (user.sellerId) {
          endpoint = `${URL}/add-to-carts/seller/${user.sellerId}`;
        } else {
          return;
        }

        const response = await axios.get(endpoint);
        const isSaved = response.data.some(
          (item) => item.fkPostpropertyId === property.postpropertyId
        );
        setSaved(isSaved);
      } catch (error) {
        console.error("Error checking saved status:", error);
      }
    };

    checkIfSaved();
  }, [user, property]);

  // Fetch current cart count
  const fetchCartCount = async () => {
    try {
      const userId = user?.id || 
                     user?.userId || 
                     user?.sellerId || 
                     user?.buyerId ||
                     user?.fkBuyerIdSellerId ||
                     user?.fkbuyerId ||
                     user?.user_id;

      if (!userId) return;

      const response = await axios.get(`${URL}/add-to-carts/user/${userId}/count`);
      
      if (response.data && response.data.count !== undefined) {
        setCartCount(response.data.count);
      }
    } catch (error) {
      console.error("Error fetching cart count:", error);
      setCartCount(0);
    }
  };

  // Check if user can add more properties to cart
  const canAddToCart = () => {
    if (!activePackage) {
      return false;
    }
    const cartLimit = activePackage.propertyAddToCart || 0;
    return cartCount < cartLimit;
  };

  // Show immediate notification
  const showImmediateNotification = (type, reason) => {
    if (type === "no_package") {
      // Toast message
      message.error({
        content: "❌ No Active Package! Purchase a package to save properties.",
        duration: 4,
        style: {
          marginTop: '20vh',
          fontSize: '16px',
          fontWeight: 600,
        },
      });

      // Desktop notification
      notification.error({
        message: '🚫 Cannot Save Property',
        description: 'You need an active membership package to save properties. Click to purchase now!',
        duration: 6,
        placement: 'topRight',
        onClick: () => {
          navigate("/membership");
        },
        style: {
          cursor: 'pointer',
          backgroundColor: '#fff2f0',
          border: '2px solid #ff4d4f',
        },
      });
    } else if (type === "limit_reached") {
      const cartLimit = activePackage?.propertyAddToCart || 0;
      
      // Toast message
      message.error({
        content: `🚫 Cart Limit Full! (${cartCount}/${cartLimit}) - Upgrade your package!`,
        duration: 5,
        style: {
          marginTop: '20vh',
          fontSize: '16px',
          fontWeight: 600,
        },
      });

      // Desktop notification
      notification.error({
        message: '⚠️ Cart Limit Reached!',
        description: `You've saved ${cartCount} out of ${cartLimit} properties. Upgrade your package to save more!`,
        duration: 6,
        placement: 'topRight',
        onClick: () => {
          navigate("/membership");
        },
        style: {
          cursor: 'pointer',
          backgroundColor: '#fff2f0',
          border: '2px solid #ff4d4f',
        },
      });
    }
  };

  // Show upgrade modal
  const showUpgradeModal = (reason) => {
    let title = "";
    let content = "";
    let icon = null;

    if (reason === "no_package") {
      title = "❌ No Active Package";
      icon = <CloseCircleOutlined style={{ color: "#ff4d4f", fontSize: 24 }} />;
      content = (
        <div>
          <Alert
            message="Package Required!"
            description="You don't have an active membership package. To save properties and access premium features, please purchase a membership package."
            type="error"
            showIcon
            style={{ marginBottom: 16, fontSize: '14px' }}
          />
          <Text strong style={{ fontSize: "15px", display: 'block', marginBottom: 8 }}>
            🎯 Benefits of Lello Prime:
          </Text>
          <ul style={{ marginTop: 8, fontSize: "14px", lineHeight: '1.8' }}>
            <li>✅ Save multiple properties to your cart</li>
            <li>✅ Direct contact with property owners</li>
            <li>✅ Zero commission fees</li>
            <li>✅ Priority customer support</li>
            <li>✅ Exclusive property deals</li>
          </ul>
          <div style={{ 
            marginTop: 16, 
            padding: '12px', 
            backgroundColor: '#e6f7ff', 
            borderLeft: '4px solid #1890ff',
            borderRadius: '4px'
          }}>
            <Text style={{ fontSize: '13px' }}>
              💡 <strong>Get started:</strong> Choose a package that fits your needs and start saving properties today!
            </Text>
          </div>
        </div>
      );
    } else if (reason === "limit_reached") {
      const cartLimit = activePackage?.propertyAddToCart || 0;
      title = "🚫 Cart Limit Full!";
      icon = <WarningOutlined style={{ color: "#ff4d4f", fontSize: 24 }} />;
      content = (
        <div>
          <Alert
            message={`Cart Full - ${cartCount}/${cartLimit} Properties`}
            description={`You've reached your cart limit of ${cartLimit} properties. Upgrade your package to save more properties and get additional benefits.`}
            type="error"
            showIcon
            style={{ marginBottom: 16, fontSize: '14px' }}
          />
          <Text strong style={{ fontSize: "15px", display: 'block', marginBottom: 8 }}>
            📊 Current Package Status:
          </Text>
          <ul style={{ marginTop: 8, fontSize: "14px", lineHeight: '1.8' }}>
            <li><strong>Package:</strong> {activePackage?.packageName}</li>
            <li><strong>Cart Limit:</strong> {cartLimit} properties</li>
            <li><strong>Properties Saved:</strong> {cartCount} / {cartLimit}</li>
            <li style={{ color: "#ff4d4f", fontWeight: 600 }}>❌ Available Slots: 0 (LIMIT FULL!)</li>
          </ul>
          <Divider style={{ margin: "12px 0" }} />
          <div style={{ 
            padding: '12px', 
            backgroundColor: '#fff7e6', 
            borderLeft: '4px solid #faad14',
            borderRadius: '4px'
          }}>
            <Text style={{ fontSize: '13px' }}>
              💡 <strong>Solution:</strong> Upgrade to a higher package to increase your cart limit and save more properties!
            </Text>
          </div>
        </div>
      );
    }

    Modal.confirm({
      title: (
        <div style={{ fontSize: '18px', fontWeight: 700, color: '#000' }}>
          {title}
        </div>
      ),
      icon: icon,
      content: content,
      okText: reason === "no_package" ? "🛒 Purchase Package Now" : "⬆️ Upgrade Package Now",
      cancelText: "Cancel",
      okButtonProps: {
        danger: true,
        size: "large",
        style: {
          fontWeight: 600,
          height: '42px',
          fontSize: '15px',
        }
      },
      cancelButtonProps: {
        size: "large",
        style: {
          height: '42px',
        }
      },
      onOk: () => {
        navigate("/membership");
      },
      width: 600,
      centered: true,
      maskClosable: false,
    });
  };

  const toggleSave = async () => {
    if (!user) {
      message.warning("Please login to save properties");
      return;
    }

    if (!property) {
      message.error("Property data not available");
      return;
    }

    // If trying to unsave, allow it
    if (saved) {
      setLoading(true);
      try {
        let endpoint;
        if (user.buyerId) {
          endpoint = `${URL}/add-to-carts/buyer/${user.buyerId}`;
        } else if (user.sellerId) {
          endpoint = `${URL}/add-to-carts/seller/${user.sellerId}`;
        } else {
          message.error("User type not found");
          return;
        }

        const response = await axios.get(endpoint);
        
        const cartItem = response.data.find(
          (item) => item.fkPostpropertyId === property.postpropertyId
        );

        if (cartItem) {
          await axios.delete(`${URL}/add-to-carts/${cartItem.addTocartId}`);
          setSaved(false);
          message.success("Property removed from saved list");
          await fetchCartCount();
        }
      } catch (error) {
        console.error("Error removing property:", error);
        message.error("Failed to update saved status. Please try again.");
      } finally {
        setLoading(false);
      }
      return;
    }

    // If trying to save, check limits
    setCheckingLimit(true);

    // Check if user has active package
    if (!activePackage) {
      setCheckingLimit(false);
      // Show immediate notification
      showImmediateNotification("no_package");
      // Then show detailed modal after short delay
      setTimeout(() => showUpgradeModal("no_package"), 600);
      return;
    }

    // Check if user has reached cart limit
    if (!canAddToCart()) {
      setCheckingLimit(false);
      // Show immediate notification
      showImmediateNotification("limit_reached");
      // Then show detailed modal after short delay
      setTimeout(() => showUpgradeModal("limit_reached"), 600);
      return;
    }

    // User can save the property
    setLoading(true);
    setCheckingLimit(false);

    try {
      const now = new Date();
      const usertypeId = user.fkusertypeId || user.fkUsertypeId || user.usertypeId || user.fkuserId;
      const buyerSellerId = user.buyerId || user.sellerId;
      
      if (!buyerSellerId) {
        message.error("User ID not found. Please login again.");
        return;
      }

      const cartData = {
        fkBuyerIdSellerId: buyerSellerId,
        fkPostpropertyId: property.postpropertyId,
        fkUsertypeId: usertypeId,
        date: now.toISOString(),
        time: now.toTimeString().split(" ")[0],
        lastChanged: now.toISOString(),
      };

      await axios.post(`${URL}/add-to-carts`, cartData);
      setSaved(true);
      
      // Success notification with cart status
      const newCount = cartCount + 1;
      const cartLimit = activePackage.propertyAddToCart || 0;
      
      message.success({
        content: `✅ Property saved successfully! (${newCount}/${cartLimit})`,
        duration: 3,
      });
      
      // Update cart count
      await fetchCartCount();
      
      // Show warning if approaching limit
      if (newCount >= cartLimit * 0.8 && newCount < cartLimit) {
        notification.warning({
          message: '⚠️ Cart Almost Full',
          description: `You have ${cartLimit - newCount} slots remaining. Consider upgrading soon!`,
          duration: 5,
          placement: 'topRight',
        });
      }
    } catch (error) {
      console.error("Error toggling save:", error);
      message.error(
        error.response?.data?.error?.message || 
        "Failed to update saved status. Please try again."
      );
    } finally {
      setLoading(false);
    }
  };

  // Get photos array
  const photos = property?.photos || [];
  const mainPhoto = photos.length > 0 ? photos[0] : property?.photo;
  const gridPhotos = photos.slice(1, 5);

  // Fill empty slots
  const fillEmptySlots = () => {
    const slots = [];
    const remainingSlots = 4 - gridPhotos.length;

    for (let i = 0; i < remainingSlots; i++) {
      slots.push(
        <Col span={24} key={`default-${i}`}>
          <Image
            width="100%"
            height={200}
            src={img2}
            alt="Property"
            style={{ objectFit: "cover", borderRadius: "8px" }}
          />
        </Col>
      );
    }
    return slots;
  };

  // Format Price
  const formatPrice = (price) => {
    if (!price) return "N/A";
    const num = Number(price);

    if (num >= 10000000) {
      return (num / 10000000).toFixed(2).replace(/\.00$/, "") + " Cr";
    } else if (num >= 100000) {
      return (num / 100000).toFixed(2).replace(/\.00$/, "") + " Lakh";
    } else {
      return num.toLocaleString("en-IN");
    }
  };

  return (
    <div
      style={{
        padding: "24px 24px 0px 24px",
        maxWidth: "1500px",
        margin: "auto",
      }}
    >
      {/* Image Section */}
      <Row gutter={[16, 16]} style={{ paddingBottom: 20 }}>
        <Col xs={24} md={12}>
          <Image
            width="100%"
            height={410}
            src={mainPhoto}
            alt="Main property"
            style={{ objectFit: "cover", borderRadius: "12px" }}
          />
        </Col>
        <Col xs={24} md={6}>
          <Row gutter={[8, 8]}>
            {gridPhotos.slice(0, 2).map((photo, index) => (
              <Col span={24} key={index}>
                <Image
                  width="100%"
                  height={200}
                  src={photo}
                  alt={`Property ${index + 1}`}
                  style={{ objectFit: "cover", borderRadius: "8px" }}
                />
              </Col>
            ))}
            {gridPhotos.length < 2 &&
              fillEmptySlots().slice(0, 2 - gridPhotos.length)}
          </Row>
        </Col>
        <Col xs={24} md={6}>
          <Row gutter={[8, 8]}>
            {gridPhotos.slice(2, 4).map((photo, index) => (
              <Col span={24} key={index + 2}>
                <Image
                  width="100%"
                  height={200}
                  src={photo}
                  alt={`Property ${index + 3}`}
                  style={{ objectFit: "cover", borderRadius: "8px" }}
                />
              </Col>
            ))}
            {gridPhotos.length < 4 &&
              fillEmptySlots().slice(Math.max(0, 2 - gridPhotos.length))}
          </Row>
        </Col>
      </Row>

      {/* Title and Info */}
      <Row
        justify="space-between"
        align="middle"
        style={{ backgroundColor: "white", padding: 20, borderRadius: 16 }}
      >
        <Col>
          <Row gutter={16} align="middle">
            <Col>
              <Title
                level={3}
                style={{ marginBottom: 0, fontFamily: "Poppins" }}
              >
                ₹{formatPrice(property?.price)}
              </Title>
            </Col>
            <Divider
              type="vertical"
              style={{
                height: "50px",
                margin: "0 10px",
                borderColor: "#d9d9d9",
              }}
            />

            <Col>
              <Title
                level={3}
                style={{ marginBottom: 0, fontFamily: "Poppins" }}
              >
                {property?.Property_Title || "Property"}
              </Title>
              <Text type="secondary" style={{ fontFamily: "Poppins" }}>
                {property?.sublocality ||
                  `${property?.fklocalityId}, ${property?.fkcityId}`}
              </Text>
            </Col>
          </Row>
          <Row gutter={16} align="middle">
            <Col>
              <div style={{ marginTop: 8 }}>
                {property?.RERA_is_active === 1 && (
                  <Tag color="green" style={{ fontFamily: "Poppins" }}>
                    RERA
                  </Tag>
                )}
                <Tag color="cyan" style={{ fontFamily: "Poppins" }}>
                  Lello Certified Property
                </Tag>
                <Tag color="blue" style={{ fontFamily: "Poppins" }}>
                  8 Top Facilities
                </Tag>
              </div>
            </Col>
          </Row>
        </Col>

        <Col>
          <Row gutter={12} align="middle">
            <Col>
              <Button
                size="large"
                onClick={toggleSave}
                loading={loading || checkingLimit}
                icon={saved ? <HeartFilled /> : <HeartOutlined />}
                style={{
                  borderRadius: 30,
                  padding: "0 22px",
                  fontFamily: "Poppins",
                  borderColor: saved ? "#ff4d4f" : "#d9d9d9",
                  color: saved ? "#ff4d4f" : "#444",
                  background: "#fff",
                }}
              >
                {checkingLimit ? "Checking..." : saved ? "Saved" : "Save"}
              </Button>
            </Col>

            <Col>
              <Button
                type="primary"
                icon={<PhoneOutlined />}
                size="large"
                style={{
                  borderRadius: 30,
                  padding: "0 26px",
                  fontFamily: "Poppins",
                }}
              >
                View Number
              </Button>
            </Col>
          </Row>
        </Col>
      </Row>

      {/* Only show alert banner when limit is actually reached (100%) */}
      {activePackage && cartCount >= (activePackage.propertyAddToCart || 0) && (
        <Alert
          message="🚫 Cart Limit Reached!"
          description={`You've saved ${cartCount} out of ${activePackage.propertyAddToCart || 0} properties. Your cart is full - please upgrade to save more.`}
          type="error"
          showIcon
          style={{ marginTop: 16, borderRadius: 8 }}
          action={
            <Button 
              size="small" 
              type="primary"
              danger
              onClick={() => navigate("/membership")}
            >
              Upgrade Now
            </Button>
          }
        />
      )}

      {/* Construction Status */}
      <Card
        style={{ marginTop: 24, backgroundColor: "#3C93FD", border: "none" }}
        bodyStyle={{ padding: "16px" }}
      >
        <Text
          type="secondary"
          style={{ fontSize: "18px", color: "white", fontFamily: "Poppins" }}
        >
          CONSTRUCTION STATUS
        </Text>
        <br />
        <Text style={{ fontFamily: "Poppins", color: "white" }}>
          Completion in Dec, 2028
        </Text>
      </Card>
    </div>
  );
};

export default PropertyPhotos;