import React, { useState, useEffect, useRef } from "react";
import { useNavigate } from "react-router-dom";
import { FiFilter } from "react-icons/fi";
import { Button, Select, Checkbox, Drawer, message } from "antd";
import SearchIcon from "@mui/icons-material/Search";
import CheckIcon from "@mui/icons-material/Check";
import ClearIcon from "@mui/icons-material/Clear";
import { URL } from "../../../config";

const SearchBarNew = ({ onFilterToggle }) => {
  const navigate = useNavigate();
  const [showFilters, setShowFilters] = useState(false);
  const [activeTab, setActiveTab] = useState("");
  const [activeTabId, setActiveTabId] = useState(null); // Store tab ID
  const [searchCity, setSearchCity] = useState("");
  const [budget, setBudget] = useState(undefined);

  const [bedrooms, setBedrooms] = useState(undefined);
  const [furnishing, setFurnishing] = useState(undefined);
  const [propertyTypes, setPropertyTypes] = useState([]);
  const [propertyTypesList, setPropertyTypesList] = useState([]);
  const [mainPropertyTypesList, setMainPropertyTypesList] = useState([]);
  const [mainPropertyType, setMainPropertyType] = useState(undefined);
  const [userTypes, setUserTypes] = useState([]);
  const [selectedUserType, setSelectedUserType] = useState(undefined);

  const [tabs, setTabs] = useState([]);

  const filterRef = useRef(null);
  const filterContainerRef = useRef(null);

  useEffect(() => {
    const handleClickOutside = (event) => {
      // Ignore clicks inside Ant Design dropdowns or popups
      if (
        event.target.closest(".ant-select-dropdown") ||
        event.target.closest(".ant-picker-dropdown") ||
        event.target.closest(".ant-dropdown") ||
        event.target.closest(".ant-modal")
      ) {
        return;
      }

      if (
        filterContainerRef.current &&
        !filterContainerRef.current.contains(event.target)
      ) {
        setShowFilters(false);
        if (onFilterToggle) onFilterToggle(false);
      }
    };

    if (showFilters) {
      document.addEventListener("mousedown", handleClickOutside);
    } else {
      document.removeEventListener("mousedown", handleClickOutside);
    }

    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, [showFilters, onFilterToggle]);

  useEffect(() => {
    const fetchTabs = async () => {
      try {
        const response = await fetch(`${URL}/post-type-masters`);
        const data = await response.json();
        if (Array.isArray(data)) {
          setTabs(data);
          if (data.length > 0) {
            setActiveTab(data[0]?.posttypeName || "");
            setActiveTabId(data[0]?.postTypeId || null);
          }
        }
      } catch (error) {
        console.error("Error fetching post types:", error);
      }
    };
    fetchTabs();
  }, []);

  useEffect(() => {
    const fetchPropertyTypes = async () => {
      try {
        const response = await fetch(`${URL}/property-type-masters`);
        const data = await response.json();
        if (Array.isArray(data)) {
          setPropertyTypesList(data);
        }
      } catch (error) {
        console.error("Error fetching property types:", error);
      }
    };
    fetchPropertyTypes();
  }, []);

  useEffect(() => {
    const fetchMainPropertyTypes = async () => {
      try {
        const response = await fetch(`${URL}/main-propertty-type-masters`);
        const data = await response.json();
        if (Array.isArray(data)) {
          setMainPropertyTypesList(data);
        }
      } catch (error) {
        console.error("Error fetching main property types:", error);
      }
    };
    fetchMainPropertyTypes();
  }, []);

  useEffect(() => {
    const fetchUserTypes = async () => {
      try {
        const response = await fetch(`${URL}/usertypes`);
        const data = await response.json();
        if (Array.isArray(data)) {
          // Exclude userId = 1
          const filtered = data.filter((u) => u.userId !== 1);
          setUserTypes(filtered);
        }
      } catch (error) {
        console.error("Error fetching user types:", error);
      }
    };
    fetchUserTypes();
  }, []);

  // Updated handleSearch function with IDs
  // const handleSearch = () => {
  //   const searchParams = new URLSearchParams();

  //   // Add search query
  //   if (searchCity.trim()) {
  //     searchParams.append("search", searchCity.trim());
  //   }

  //   // Add active tab ID (post type)
  //   if (activeTabId !== null) {
  //     searchParams.append("postTypeId", activeTabId);
  //   }

  //   // Add other filters if applied
  //   if (budget) searchParams.append("budget", budget);

  //   // Main Property Type - send ID
  //   if (mainPropertyType) {
  //     searchParams.append("mainPropertyTypeId", mainPropertyType);
  //   }

  //   if (selectedUserType) {
  //     searchParams.append("userId", selectedUserType);
  //   }

  //   if (bedrooms) searchParams.append("bedrooms", bedrooms);
  //   if (furnishing) searchParams.append("furnishing", furnishing);

  //   // Property Types - send IDs as comma-separated
  //   if (propertyTypes.length > 0) {
  //     searchParams.append("propertyTypeIds", propertyTypes.join(","));
  //   }

  //   // Navigate to property listing page with search params
  //   navigate(`/property-listing?${searchParams.toString()}`);
  // };

  const handleClear = () => {
    setBudget(undefined);
    setMainPropertyType(undefined);
    setBedrooms(undefined);
    setFurnishing(undefined);
    setPropertyTypes([]);
    // Reset to first tab
    if (tabs.length > 0) {
      setActiveTab(tabs[0]?.posttypeName || "");
      setActiveTabId(tabs[0]?.postTypeId || null);
    }
    setSearchCity("");
  };

  // In SearchBarNew.js, update the FilterContent component and handle functions

const FilterContent = () => (
  <div>
    <div      
      style={{
        display: "grid",
        gridTemplateColumns: "repeat(auto-fit, minmax(100px, 1fr))",
        gap: "20px",
        marginBottom: "24px",
      }}
    >
      <div>
        <label
          style={{
            display: "block",
            marginBottom: "8px",
            fontWeight: 500,
            color: "#1f2937",
            fontSize: "14px",
            fontFamily: "Poppins",
          }}
        >
          Budget
        </label>
        <Select
          placeholder="Select Budget"
          value={budget}
          onChange={setBudget}
          size="large"
          style={{ width: "100%" }}
          allowClear
          options={[
            { value: "0-50", label: "₹0 - 50 Lakhs" },
            { value: "50-1", label: "₹50L - 1Cr" },
            { value: "1-2", label: "₹1Cr - 2Cr" },
            { value: "2+", label: "₹2Cr+" },
          ]}
        />
      </div>

      <div>
        <label
          style={{
            display: "block",
            marginBottom: "8px",
            fontWeight: 500,
            color: "#1f2937",
            fontSize: "14px",
            fontFamily: "Poppins",
          }}
        >
          Main Property Type
        </label>
        <Select
          placeholder="Select Main Property Type"
          value={mainPropertyType}
          onChange={setMainPropertyType}
          size="large"
          style={{ width: "100%" }}
          allowClear
          showSearch
          options={mainPropertyTypesList.map((item) => ({
            value: item.mainpropertytypeId, // Use ID as value
            label: item.mainprpoertyName,
          }))}
        />
      </div>

      <div>
        <label
          style={{
            display: "block",
            marginBottom: "8px",
            fontWeight: 500,
            color: "#1f2937",
            fontSize: "14px",
            fontFamily: "Poppins",
          }}
        >
          Posted By
        </label>
        <Select
          placeholder="Select User Type"
          value={selectedUserType}
          onChange={setSelectedUserType}
          size="large"
          style={{ width: "100%" }}
          allowClear
          options={userTypes.map((user) => ({
            value: user.userId,
            label: user.username,
          }))}
        />
      </div>

      <div>
        <label
          style={{
            display: "block",
            marginBottom: "8px",
            fontWeight: 500,
            color: "#1f2937",
            fontSize: "14px",
            fontFamily: "Poppins",
          }}
        >
          Bedrooms
        </label>
        <Select
          placeholder="Select Bedrooms"
          value={bedrooms}
          onChange={setBedrooms}
          size="large"
          style={{ width: "100%" }}
          allowClear
          options={[
            { value: "1", label: "1 BHK" },
            { value: "2", label: "2 BHK" },
            { value: "3", label: "3 BHK" },
            { value: "4", label: "4 BHK" },
            { value: "5", label: "5+ BHK" },
          ]}
        />
      </div>

      <div>
        <label
          style={{
            display: "block",
            marginBottom: "8px",
            fontWeight: 500,
            color: "#1f2937",
            fontSize: "14px",
            fontFamily: "Poppins",
          }}
        >
          Furnishing
        </label>
        <Select
          placeholder="Select Type"
          value={furnishing}
          onChange={setFurnishing}
          size="large"
          style={{ width: "100%" }}
          allowClear
          options={[
            { value: "furnished", label: "Furnished" },
            { value: "semi-furnished", label: "Semi-Furnished" },
            { value: "unfurnished", label: "Unfurnished" },
          ]}
        />
      </div>
    </div>

     <div style={{ marginBottom: "24px" }}>
      <h3
        style={{
          marginBottom: "14px",
          color: "#1f2937",
          fontSize: "15px",
          fontWeight: 600,
          fontFamily: "Poppins",
        }}
      >
        Property Types
      </h3>
      <div
        style={{
          display: "grid",
          gridTemplateColumns: "repeat(auto-fill, minmax(200px, 1fr))",
          gap: "12px",
        }}
      >
        {propertyTypesList.map((type) => (
          <Checkbox
            key={type.propertyTypeId}
            checked={propertyTypes.includes(type.propertyTypeId)}
            onChange={(e) => {
              if (e.target.checked) {
                setPropertyTypes([...propertyTypes, type.propertyTypeId]);
              } else {
                setPropertyTypes(
                  propertyTypes.filter((t) => t !== type.propertyTypeId)
                );
              }
            }}
            style={{
              fontSize: "14px",
              fontFamily: "Poppins",
            }}
          >
            {type.propertytypeName}
          </Checkbox>
        ))}
      </div>
    </div>

    <div
      style={{
        display: "flex",
        gap: "12px",
        justifyContent: "flex-end",
        paddingTop: "20px",
        borderTop: "1px solid #e5e7eb",
        flexWrap: "wrap",
      }}
    >
      <Button
        icon={<ClearIcon />}
        size="large"
        onClick={handleClear}
        style={{
          fontWeight: 500,
          fontFamily: "Poppins",
          borderRadius: "8px",
          minWidth: "130px",
          height: "44px",
        }}
      >
        Clear All
      </Button>
      <Button
        type="primary"
        icon={<CheckIcon />}
        size="large"
        onClick={() => {
          // Apply filters but don't navigate yet
          const newState = !showFilters;
          setShowFilters(newState);
          if (onFilterToggle) onFilterToggle(newState);
        }}
        style={{
          fontWeight: 600,
          fontFamily: "Poppins",
          backgroundColor: "#1e40af",
          borderColor: "#1e40af",
          borderRadius: "8px",
          minWidth: "130px",
          height: "44px",
          boxShadow: "0 4px 12px rgba(30, 64, 175, 0.25)",
        }}
      >
        Apply Filters
      </Button>
    </div>
  </div>
);

// Update the handleSearch function
// Update the handleSearch function
const handleSearch = () => {
  if (!searchCity.trim()) {
    message.warning("Please enter a search term");
    return;
  }

  const searchParams = new URLSearchParams();

  // Add search query
  if (searchCity.trim()) {
    searchParams.append("search", searchCity.trim());
  }

  // Add active tab ID (post type)
  if (activeTabId !== null) {
    searchParams.append("postTypeId", activeTabId);
  }

  // Add other filters if applied
  if (budget) searchParams.append("budget", budget);

  // Main Property Type - send ID
  if (mainPropertyType) {
    searchParams.append("mainPropertyTypeId", mainPropertyType);
  }

  if (selectedUserType) {
    searchParams.append("userId", selectedUserType);
  }

  if (bedrooms) searchParams.append("bedrooms", bedrooms);
  if (furnishing) searchParams.append("furnishing", furnishing);

   // Property Types - send IDs as comma-separated (FIXED)
  if (propertyTypes.length > 0) {
    searchParams.append("propertyTypeIds", propertyTypes.join(","));
  }

  // Navigate to property listing page with search params
  navigate(`/property-listing?${searchParams.toString()}`);
};

  return (
    <>
      <style>
        {`
          @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&family=Poppins:wght@400;500;600;700&display=swap');
          
          .tab-button {
            position: relative;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
          }

          .tab-button::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 50%;
            transform: translateX(-50%) scaleX(0);
            width: 100%;
            height: 3px;
            background: linear-gradient(90deg, #1e40af 0%, #3b82f6 100%);
            border-radius: 3px 3px 0 0;
            transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
          }

          .tab-button.active::after {
            transform: translateX(-50%) scaleX(1);
          }

          .tab-button:hover:not(.active) {
            color: #1e40af;
            background: rgba(30, 64, 175, 0.05);
          }

          .tabs-scroll-container {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            scrollbar-width: thin;
            scrollbar-color: #cbd5e1 #f1f5f9;
          }

          .tabs-scroll-container::-webkit-scrollbar {
            height: 6px;
          }

          .tabs-scroll-container::-webkit-scrollbar-track {
            background: #f1f5f9;
            border-radius: 10px;
          }

          .tabs-scroll-container::-webkit-scrollbar-thumb {
            background: #cbd5e1;
            border-radius: 10px;
          }

          .tabs-wrapper {
            display: inline-flex;
            flex-wrap: nowrap;
            min-width: max-content;
            gap: 4px;
          }

          @media (max-width: 992px) {
            .tabs-scroll-container {
              overflow-x: auto;
              -webkit-overflow-scrolling: touch;
              scrollbar-width: thin;
              scrollbar-color: #cbd5e1 #f1f5f9;
            }
            .tabs-scroll-container::-webkit-scrollbar {
              height: 4px;
            }
            .tabs-scroll-container::-webkit-scrollbar-track {
              background: #f1f5f9;
              border-radius: 10px;
            }
            .tabs-scroll-container::-webkit-scrollbar-thumb {
              background: #cbd5e1;
              border-radius: 10px;
            }
            .tabs-wrapper {
              flex-wrap: nowrap !important;
              min-width: max-content;
            }
          }

          @media (max-width: 768px) {
            .search-container {
              padding: 16px !important;
            }
            .search-bar {
              border-radius: 40px !important;
              padding: 5px !important;
            }
            .search-input {
              font-size: 14px !important;
              padding: 10px 14px !important;
            }
            .search-btn {
              padding: 0 24px !important;
              height: 42px !important;
              font-size: 14px !important;
            }
            .filter-icon-wrapper {
              padding: 11px !important;
              width: 42px !important;
              height: 42px !important;
            }
            .tab-button {
              font-size: 13px !important;
              padding: 12px 20px !important;
            }
            .tabs-container {
              padding: 16px 16px 0 16px !important;
            }
          }

          @media (max-width: 576px) {
            .search-input {
              font-size: 13px !important;
              padding: 8px 12px !important;
            }
            .search-btn span {
              display: none;
            }
            .tab-button {
              font-size: 12px !important;
              padding: 10px 18px !important;
            }
          }
        `}
      </style>

      <div
        style={{
          width: "100%",
          maxWidth: "1100px",
          margin: "0 auto",
        }}
      >
        {/* Tabs Section */}
        <div
          className="tabs-container"
          style={{
            padding: "10px 10px 0 10px",
            backgroundColor: "#ffffff",
            borderRadius: "16px 16px 0 0",
            boxShadow: "0 1px 3px rgba(0, 0, 0, 0.05)",
          }}
        >
          <div
            className="tabs-scroll-container"
            style={{ paddingBottom: "4px" }}
          >
            <div
              className="tabs-wrapper"
              style={{
                display: "flex",
                gap: "4px",
                flexWrap: "wrap",
                borderBottom: "2px solid #f1f5f9",
              }}
            >
              {tabs.map((tab) => (
                <button
                  key={tab.postTypeId}
                  onClick={() => {
                    setActiveTab(tab.posttypeName);
                    setActiveTabId(tab.postTypeId); // Store ID when tab changes
                  }}
                  className={`tab-button ${
                    activeTab === tab.posttypeName ? "active" : ""
                  }`}
                  style={{
                    padding: "14px 28px",
                    fontSize: "14px",
                    fontWeight: activeTab === tab.posttypeName ? 600 : 500,
                    fontFamily: "Inter, Poppins, sans-serif",
                    color:
                      activeTab === tab.posttypeName ? "#1e40af" : "#64748b",
                    backgroundColor: "transparent",
                    border: "none",
                    cursor: "pointer",
                    outline: "none",
                    whiteSpace: "nowrap",
                    borderRadius: "8px 8px 0 0",
                    letterSpacing: "-0.01em",
                  }}
                >
                  {tab.posttypeName}
                </button>
              ))}
            </div>
          </div>
        </div>

        {/* Search Bar Container */}
        <div
          className="search-container"
          style={{
            padding: "10px",
            backgroundColor: "#ffffff",
            borderRadius: "0 0 16px 16px",
            boxShadow:
              "0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06)",
          }}
        >
          {/* Main Search Bar */}
          <div
            className="search-bar"
            style={{
              display: "flex",
              alignItems: "center",
              backgroundColor: "#ffffff",
              borderRadius: "50px",
              padding: "6px",
              boxShadow:
                "0 0 0 1px rgba(0, 0, 0, 0.05), 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05)",
              border: "none",
              transition: "box-shadow 0.3s ease",
            }}
            onMouseEnter={(e) => {
              e.currentTarget.style.boxShadow =
                "0 0 0 1px rgba(30, 64, 175, 0.1), 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04)";
            }}
            onMouseLeave={(e) => {
              e.currentTarget.style.boxShadow =
                "0 0 0 1px rgba(0, 0, 0, 0.05), 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05)";
            }}
          >
            {/* Filter Icon */}
            <div
              className="filter-icon-wrapper"
              onClick={() => setShowFilters(!showFilters)}
              ref={filterRef}
              style={{
                cursor: "pointer",
                padding: "13px",
                borderRadius: "50%",
                backgroundColor: showFilters ? "#1e40af" : "#eff6ff",
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                marginLeft: "6px",
                transition: "all 0.3s ease",
                flexShrink: 0,
                width: "48px",
                height: "48px",
              }}
            >
              <FiFilter
                size={20}
                color={showFilters ? "white" : "#1e40af"}
                strokeWidth={2.5}
              />
            </div>

            {/* Search Input */}
            <input
              type="text"
              placeholder="Search by Property Title or City Name or Locality"
              value={searchCity}
              onChange={(e) => setSearchCity(e.target.value)}
              onKeyPress={(e) => {
                if (e.key === "Enter") {
                  handleSearch();
                }
              }}
              className="search-input"
              style={{
                flex: 1,
                fontSize: "15px",
                color: "#1f2937",
                backgroundColor: "transparent",
                border: "none",
                outline: "none",
                padding: "14px 20px",
                fontFamily: "Poppins",
                minWidth: "0",
                fontWeight: 400,
                letterSpacing: "-0.01em",
              }}
            />

            {/* Search Button */}
            <Button
              type="primary"
              icon={<SearchIcon style={{ fontSize: "20px" }} />}
              size="large"
              onClick={handleSearch}
              className="search-btn"
              style={{
                fontWeight: 600,
                fontFamily: "Inter, Poppins, sans-serif",
                backgroundColor: "#1e40af",
                borderColor: "#1e40af",
                borderRadius: "50px",
                padding: "0 32px",
                height: "48px",
                display: "flex",
                alignItems: "center",
                gap: "8px",
                boxShadow: "0 4px 14px rgba(30, 64, 175, 0.3)",
                flexShrink: 0,
                transition: "all 0.3s ease",
                letterSpacing: "-0.01em",
              }}
              onMouseEnter={(e) => {
                e.currentTarget.style.backgroundColor = "#1e3a8a";
                e.currentTarget.style.transform = "translateY(-1px)";
                e.currentTarget.style.boxShadow =
                  "0 6px 20px rgba(30, 64, 175, 0.4)";
              }}
              onMouseLeave={(e) => {
                e.currentTarget.style.backgroundColor = "#1e40af";
                e.currentTarget.style.transform = "translateY(0)";
                e.currentTarget.style.boxShadow =
                  "0 4px 14px rgba(30, 64, 175, 0.3)";
              }}
            >
              <span>Search</span>
            </Button>
          </div>

          {/* Desktop Filter Panel */}
          {showFilters &&
            typeof window !== "undefined" &&
            window.innerWidth > 768 && (
              <div
                ref={filterContainerRef}
                style={{
                  marginTop: "24px",
                  paddingTop: "24px",
                  borderTop: "1px solid #f1f5f9",
                }}
              >
                <FilterContent />
              </div>
            )}
        </div>

        {/* Mobile Filter Drawer */}
        <Drawer
          title={
            <span
              style={{
                fontFamily: "Inter, Poppins",
                fontWeight: 600,
                fontSize: "18px",
                color: "#1f2937",
              }}
            >
              Advanced Filters
            </span>
          }
          placement="bottom"
          onClose={() => setShowFilters(false)}
          open={
            showFilters &&
            typeof window !== "undefined" &&
            window.innerWidth <= 768
          }
          height="90vh"
          style={{
            borderTopLeftRadius: "20px",
            borderTopRightRadius: "20px",
          }}
        >
          <FilterContent />
        </Drawer>
      </div>
    </>
  );
};

export default SearchBarNew;
