import { useState, useEffect, useRef } from "react";
import { useNavigate } from "react-router-dom";
import {
  Modal,
  Box,
  Typography,
  TextField,
  Button,
  Stack,
  IconButton,
} from "@mui/material";
import CloseIcon from "@mui/icons-material/Close";
import EditIcon from "@mui/icons-material/Edit";
import LoginModal from "./UserRegistration";
import axios from "axios";
import { useSnackbar } from "notistack";
import { URL } from "../../config";

const style = {
  position: "absolute",
  top: "50%",
  left: "50%",
  transform: "translate(-50%, -50%)",
  width: 360,
  bgcolor: "background.paper",
  boxShadow: 24,
  borderRadius: 2,
  p: 3,
  outline: "none",
};

export default function OTPModal({
  open,
  onClose,
  phone,
  onEditPhoneClick,
  handleOtpSuccess,
}) {
  const [loginModal, setLoginModal] = useState(false);
  const [otp, setOtp] = useState(["", "", "", ""]);
  const [timer, setTimer] = useState(120); // ✅ 2-minute countdown

  const navigate = useNavigate();
  const { enqueueSnackbar } = useSnackbar();
  const firstOtpRef = useRef(null);

  useEffect(() => {
    if (open) {
      setOtp(["", "", "", ""]);
      setTimer(120); // reset to 2 minutes
      setTimeout(() => {
        firstOtpRef.current?.focus();
      }, 100);
    }
  }, [open]);

  useEffect(() => {
    if (timer > 0) {
      const interval = setInterval(() => setTimer((t) => t - 1), 1000);
      return () => clearInterval(interval);
    }
  }, [timer]);

  // Format time as MM:SS
  const formatTime = (seconds) => {
    const m = String(Math.floor(seconds / 60)).padStart(2, "0");
    const s = String(seconds % 60).padStart(2, "0");
    return `${m}:${s}`;
  };

  const handleChange = (e, index) => {
    const value = e.target.value.replace(/\D/, "");
    if (value.length <= 1) {
      const newOtp = [...otp];
      newOtp[index] = value;
      setOtp(newOtp);

      if (value && index < 3) {
        document.getElementById(`otp-${index + 1}`).focus();
      }
    }
  };

  const handleKeyDown = (e, index) => {
    if (e.key === "Backspace" && !otp[index] && index > 0) {
      document.getElementById(`otp-${index - 1}`).focus();
    }
  };

  const handleVerify = async () => {
    const enteredOtp = otp.join("");
    if (enteredOtp.length !== 4) {
      enqueueSnackbar("⚠️ Please enter the complete 4-digit OTP.", {
        variant: "warning",
      });
      return;
    }

    try {
      const res = await axios.post(`${URL}/otp-verifications/verify`, {
        phone_number: phone,
        otp: enteredOtp,
      });

      if (!res.data.success) {
        enqueueSnackbar(`❌ ${res.data.message || "OTP Verification Failed"}`, {
          variant: "error",
        });
        return;
      }

      enqueueSnackbar("✅ OTP Verified successfully!", { variant: "success" });
      setOtp(["", "", "", ""]);

      // 🔍 Now check if user exists
      const checkRes = await axios.post(`${URL}/otp-verifications/check-user`, {
        phone_number: phone,
      });

      const userData = checkRes.data?.data;
      console.log(userData);

      if (
        !userData ||
        (!userData.sellerId && !userData.fkuserId && !userData.buyerId)
      ) {
        enqueueSnackbar("ℹ️ User not found, please complete registration.", {
          variant: "info",
        });
        setLoginModal(true); // ✅ Opens LoginModal
        return; // 🚫 Don't close OTPModal yet
      }

      // If user exists → safe to close OTP modal
      onClose();

      if (checkRes.data.source === "seller" && userData.sellerId) {
        localStorage.setItem("sellerId", userData.sellerId);
        localStorage.setItem("role", "seller");
        localStorage.setItem("userTypeId", userData.fkuserId);
        // console.log("sellerId", userData.sellerId);
        enqueueSnackbar("🛍️ Logged in as Seller.", { variant: "info" });
        navigate("/Dashboard");
      }

      if (userData.fkuserId) {
        // localStorage.setItem("userId", userData.fkuserId);
        // enqueueSnackbar("👤 Logged in as User.", { variant: "info" });
        navigate("/Dashboard");

        if (handleOtpSuccess) {
          handleOtpSuccess(userData);
        }
      }

      if (checkRes.data.source === "buyer" && userData.buyerId) {
        localStorage.setItem("user", JSON.stringify(userData));
        localStorage.setItem("buyerId", userData.buyerId);
        localStorage.setItem("userTypeId", userData.fkusertypeId);
        enqueueSnackbar("🛍️ Logged in as Buyer.", { variant: "info" });
      }

      // if (userData.fkuserId) {
      //   localStorage.setItem("buyerId", userData.fkuserId);
      //   // enqueueSnackbar("👤 Logged in as User.", { variant: "info" });
      // }
    } catch (error) {
      console.error(error);
      enqueueSnackbar("🚨 Something went wrong during OTP verification.", {
        variant: "error",
      });
    }
  };

  const handleResendOtp = async () => {
    try {
      const res = await axios.post(`${URL}/otp-verifications`, {
        phone_number: phone,
      });

      let msg = res.data.message;
      if (res.data.otp) {
        msg += ` (OTP: ${res.data.otp})`; // ⚠️ Remove in production

        enqueueSnackbar(msg, { variant: "success" });

        // ✅ Restart countdown timer after resend
        setTimer(120);
        setOtp(["", "", "", ""]); // clear previous OTP
        setTimeout(() => {
          firstOtpRef.current?.focus(); // focus again on first input
        }, 100);
      } else {
        enqueueSnackbar(`❌ ${res.data.message || "Failed to resend OTP."}`, {
          variant: "error",
        });
      }
    } catch (error) {
      console.error(error);
      enqueueSnackbar("🚨 Error while resending OTP.", { variant: "error" });
    }
  };

  return (
    <>
      {/* <Modal open={open} onClose={onClose}> */}

      <Modal
        className="otp-modal"
        open={open}
        onClose={() => {}} // Do nothing when clicking backdrop
        disableEscapeKeyDown // Prevent closing with Esc
        hideBackdrop={false} // Keep backdrop visible (optional)
      >
        <Box sx={style}>
          <Box
            display="flex"
            justifyContent="space-between"
            alignItems="center"
          >
            <Typography variant="h6" fontWeight="bold" fontSize={30}>
              Verify your number
            </Typography>
            <IconButton onClick={onClose}>
              <CloseIcon />
            </IconButton>
          </Box>

          <Box display="flex" alignItems="center">
            <Typography variant="h6" fontWeight="bold" fontSize={25}>
              +91-{phone}
            </Typography>
            <IconButton size="large" onClick={onEditPhoneClick}>
              <EditIcon fontSize="medium" color="primary" />
            </IconButton>
          </Box>

          <Typography mt={2} mb={1} pt={2}>
            Enter your 4 digit OTP
          </Typography>

          <Stack direction="row" spacing={2} justifyContent="start">
            {[0, 1, 2, 3].map((i) => (
              <TextField
                key={i}
                id={`otp-${i}`}
                value={otp[i]}
                onChange={(e) => handleChange(e, i)}
                onKeyDown={(e) => handleKeyDown(e, i)} // ✅ Add this
                inputRef={i === 0 ? firstOtpRef : null} // ✅ Assign ref to first input
                inputProps={{
                  maxLength: 1,
                  style: {
                    textAlign: "center",
                    fontSize: "18px",
                    width: "50px",
                    height: "50px",
                    padding: 0,
                  },
                }}
              />
            ))}
          </Stack>

          <Typography variant="body2" mt={2}>
            Haven't received yet?{" "}
            {timer > 0 ? (
              <Typography
                component="span"
                color="text.secondary"
                fontWeight="bold"
              >
                Resend available in {formatTime(timer)}
              </Typography>
            ) : (
              <Typography
                component="span"
                color="primary"
                fontWeight="bold"
                sx={{ cursor: "pointer" }}
                onClick={handleResendOtp}
              >
                Resend OTP
              </Typography>
            )}
          </Typography>

          <Button
            fullWidth
            variant="contained"
            onClick={handleVerify}
            sx={{
              mt: 4,
              fontWeight: "bold",
              textTransform: "none",
              fontSize: "18px",
            }}
          >
            Verify & Continue
          </Button>
        </Box>
      </Modal>

      <LoginModal
        open={loginModal}
        onClose={() => setLoginModal(false)}
        phone={phone}
      />
    </>
  );
}
