// src/Components/Modal/MembershipModal/MembershipConfirmationModal.js
import React, { useState, useEffect } from "react";
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Typography,
  Button,
  Box,
  Card,
  CardContent,
  Avatar,
  CircularProgress,
  Divider,
  Chip,
  Paper,
} from "@mui/material";
import {
  CheckCircle,
  CreditCard,
  Shield,
  Calendar,
  Info,
  Phone,
  Mail,
  User,
} from "lucide-react";

const MembershipConfirmationModal = ({
  open,
  onClose,
  selectedPlan,
  packages,
  onConfirm,
  loading = false,
}) => {
  const [planDetails, setPlanDetails] = useState(null);
  const [user, setUser] = useState(null);

  useEffect(() => {
    // Get user from localStorage
    const storedUser = localStorage.getItem("user");
    if (storedUser) {
      setUser(JSON.parse(storedUser));
    }
  }, []);

  useEffect(() => {
    if (selectedPlan && packages.length > 0) {
      const plan = packages.find((p) => p.packageId === selectedPlan);
      setPlanDetails(plan);
    }
  }, [selectedPlan, packages]);

  if (!planDetails || !user) {
    return null;
  }

  const handleConfirm = () => {
    onConfirm({
      packageId: planDetails.packageId,
      packageName: planDetails.packageName,
      price: planDetails.packageCost,
      duration: planDetails.packageduration,
      description: planDetails.packageDescription,
      propertyVisit: planDetails.propertyVisit,
      propertyAddToCart: planDetails.propertyAddToCart,
    });
  };

  return (
    <Dialog
      open={open}
      onClose={onClose}
      maxWidth="md"
      fullWidth
      PaperProps={{
        sx: {
          bgcolor: "#ffffff",
          boxShadow: "0 10px 30px rgba(0,0,0,0.1)",
          border: "1px solid rgba(0,0,0,0.05)",
          borderRadius: 3,
          overflow: "hidden",
        },
      }}
    >
      <DialogTitle
        sx={{
          bgcolor: "rgba(81, 174, 214, 0.1)",
          borderBottom: "1px solid rgba(81, 174, 214, 0.2)",
          py: 3,
          px: 3,
        }}
      >
        <Box sx={{ display: "flex", alignItems: "center", gap: 2 }}>
          <Avatar
            sx={{
              bgcolor: "#51aed6",
              width: 48,
              height: 48,
            }}
          >
            <Shield size={28} color="white" />
          </Avatar>
          <Box>
            <Typography variant="h5" sx={{ color: "#333", fontWeight: 600 }}>
              Confirm Your Membership
            </Typography>
            <Typography variant="body2" sx={{ color: "#666" }}>
              Review your package details before payment
            </Typography>
          </Box>
        </Box>
      </DialogTitle>

      <DialogContent sx={{ p: 3, bgcolor: "#f9f9f9" }}>
        <Box sx={{ display: "flex", flexDirection: "column", gap: 3 }}>
          {/* Package Information */}
          <Paper
            elevation={0}
            sx={{
              bgcolor: "#ffffff",
              border: "1px solid rgba(0,0,0,0.05)",
              borderRadius: 2,
              overflow: "hidden",
            }}
          >
            <Box
              sx={{
                bgcolor: "rgba(81, 174, 214, 0.05)",
                p: 2,
                borderBottom: "1px solid rgba(0,0,0,0.05)",
              }}
            >
              <Typography
                variant="h6"
                sx={{
                  color: "#333",
                  display: "flex",
                  alignItems: "center",
                  gap: 1,
                }}
              >
                <Shield size={20} color="#51aed6" />
                Package Details
              </Typography>
            </Box>
            <CardContent sx={{ p: 2 }}>
              <Box sx={{ display: "flex", flexDirection: "column", gap: 2 }}>
                <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
                  <Typography
                    sx={{ color: "#666", minWidth: "120px", fontSize: "14px" }}
                  >
                    Package Name:
                  </Typography>
                  <Typography sx={{ color: "#333", fontWeight: 600 }}>
                    {planDetails.packageName}
                  </Typography>
                </Box>
                <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
                  <Typography
                    sx={{ color: "#666", minWidth: "120px", fontSize: "14px" }}
                  >
                    Duration:
                  </Typography>
                  <Typography sx={{ color: "#333" }}>
                    {planDetails.packageduration} days
                  </Typography>
                </Box>
                <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
                  <Typography
                    sx={{ color: "#666", minWidth: "120px", fontSize: "14px" }}
                  >
                    Property Visits:
                  </Typography>
                  <Typography sx={{ color: "#333" }}>
                    {planDetails.propertyVisit || 0}
                  </Typography>
                </Box>
                <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
                  <Typography
                    sx={{ color: "#666", minWidth: "120px", fontSize: "14px" }}
                  >
                    Add to Cart:
                  </Typography>
                  <Typography sx={{ color: "#333" }}>
                    {planDetails.propertyAddToCart || 0}
                  </Typography>
                </Box>
              </Box>
            </CardContent>
          </Paper>

          {/* Payment Information */}
          <Paper
            elevation={0}
            sx={{
              bgcolor: "#ffffff",
              border: "1px solid rgba(0,0,0,0.05)",
              borderRadius: 2,
              overflow: "hidden",
            }}
          >
            <Box
              sx={{
                bgcolor: "rgba(244, 182, 46, 0.05)",
                p: 2,
                borderBottom: "1px solid rgba(0,0,0,0.05)",
              }}
            >
              <Typography
                variant="h6"
                sx={{
                  color: "#333",
                  display: "flex",
                  alignItems: "center",
                  gap: 1,
                }}
              >
                <CreditCard size={20} color="#f4b62e" />
                Payment Information
              </Typography>
            </Box>
            <CardContent sx={{ p: 2 }}>
              <Box
                sx={{
                  display: "flex",
                  justifyContent: "space-between",
                  alignItems: "center",
                  mb: 2,
                }}
              >
                <Typography sx={{ color: "#666", fontSize: "16px" }}>
                  Total Amount:
                </Typography>
                <Typography
                  variant="h5"
                  sx={{ color: "#f4b62e", fontWeight: 700 }}
                >
                  ₹ {planDetails.packageCost}
                </Typography>
              </Box>
              <Divider sx={{ my: 2, borderColor: "rgba(0,0,0,0.05)" }} />
              <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
                <Info size={16} color="#666" />
                <Typography sx={{ color: "#666", fontSize: "14px" }}>
                  You will be redirected to PhonePe secure payment gateway
                </Typography>
              </Box>
            </CardContent>
          </Paper>

          {/* Benefits Card */}
          <Paper
            elevation={0}
            sx={{
              bgcolor: "rgba(81, 174, 214, 0.05)",
              border: "1px solid rgba(81, 174, 214, 0.1)",
              borderRadius: 2,
              p: 2,
            }}
          >
            <Typography
              variant="subtitle2"
              sx={{ color: "#333", fontWeight: 600, mb: 1 }}
            >
              Membership Benefits:
            </Typography>
            <Box sx={{ display: "flex", flexWrap: "wrap", gap: 1 }}>
              <Chip
                label="Zero Commission"
                size="small"
                sx={{
                  bgcolor: "rgba(81, 174, 214, 0.1)",
                  color: "#51aed6",
                  fontWeight: 500,
                }}
              />
              <Chip
                label="Direct Owner Contact"
                size="small"
                sx={{
                  bgcolor: "rgba(81, 174, 214, 0.1)",
                  color: "#51aed6",
                  fontWeight: 500,
                }}
              />
              <Chip
                label="Priority Support"
                size="small"
                sx={{
                  bgcolor: "rgba(81, 174, 214, 0.1)",
                  color: "#51aed6",
                  fontWeight: 500,
                }}
              />
            </Box>
          </Paper>
        </Box>
      </DialogContent>

      <DialogActions
        sx={{
          p: 3,
          borderTop: "1px solid rgba(0,0,0,0.05)",
          gap: 2,
          bgcolor: "#ffffff",
        }}
      >
        <Button
          onClick={onClose}
          variant="outlined"
          sx={{
            color: "#666",
            borderColor: "#ddd",
            bgcolor: "transparent",
            "&:hover": {
              borderColor: "#bbb",
              bgcolor: "rgba(0,0,0,0.02)",
            },
          }}
        >
          Cancel
        </Button>
        <Button
          onClick={handleConfirm}
          variant="contained"
          disabled={loading}
          startIcon={
            loading ? (
              <CircularProgress size={20} color="inherit" />
            ) : (
              <CheckCircle size={20} />
            )
          }
          sx={{
            bgcolor: "linear-gradient(45deg, #f4b62e 0%, #e0a020 100%)",
            color: "white",
            fontWeight: 600,
            boxShadow: "0 4px 10px rgba(244, 182, 46, 0.3)",
            "&:hover": {
              bgcolor: "linear-gradient(45deg, #e0a020 0%, #c99010 100%)",
              boxShadow: "0 6px 12px rgba(244, 182, 46, 0.4)",
            },
            "&:disabled": {
              bgcolor: "rgba(244, 182, 46, 0.3)",
              color: "rgba(255,255,255,0.5)",
            },
          }}
        >
          {loading ? "Processing..." : "Confirm & Pay"}
        </Button>
      </DialogActions>
    </Dialog>
  );
};

export default MembershipConfirmationModal;